<?php

declare(strict_types=1);

namespace Drupal\user_restrictions\Entity;

use Drupal\Core\Form\FormStateInterface;
use Drupal\user_restrictions\Plugin\UserRestrictionTypeInterface;

/**
 * The interface for user restriction entities.
 */
interface UserRestrictionInterface {

  /**
   * The pattern type for allowing an entered value.
   */
  public const int PATTERN_ALLOW = 1;

  /**
   * Gets the user restriction type plugin ID.
   *
   * @return string
   *   The plugin ID.
   */
  public function getTypePluginId(): string;

  /**
   * Gets the restriction type.
   *
   * @return string
   *   The restriction type.
   *
   * @deprecated in user_restrictions:2.1.1 and is removed from
   * user_restrictions:3.0.0. Use
   * \Drupal\user_restrictions\UserRestrictionInterface::getPluginId() instead.
   *
   * @see https://www.drupal.org/node/3554615
   */
  public function getRuleType(): string;

  /**
   * Gets the pattern value.
   *
   * @return string
   *   A regular expression, or an exact string, to match against a value.
   */
  public function getPattern(): string;

  /**
   * Gets the pattern type.
   *
   * @return int
   *   The pattern type.
   */
  public function getPatternType(): int;

  /**
   * Gets the restriction type.
   *
   * @return bool
   *   TRUE when the restriction allows the data to be used, FALSE otherwise.
   *
   * @deprecated in user_restriction:2.1.1 and is removed
   *   from user_restrictions:3.0.0. Use
   *   \Drupal\user_restrictions\UserRestrictionInterface::getPatternType()
   *   instead.
   *
   * @see https://www.drupal.org/node/3548381
   */
  public function getAccessType(): bool;

  /**
   * Gets the forms to which the restriction applies.
   *
   * @return array
   *   The forms to which the restriction applies.
   */
  public function getForms(): array;

  /**
   * Gets the no_expiration value.
   *
   * @return bool
   *   The no_expiration value, TRUE when the stored expiration time is ignored,
   *   FALSE otherwise.
   */
  public function expires(): bool;

  /**
   * Gets the expiration time.
   *
   * @return int|null
   *   The expiration time.
   */
  public function getExpiration(): ?int;

  /**
   * Gets the expiration time.
   *
   * @return int|null
   *   The expiration time.
   *
   * @deprecated in user_restrictions:2.1.1 and is removed
   *   from user_restriction:3.0.0. Use
   *   \Drupal\user_restrictions\UserRestrictionInterface::getExpiration()
   *   instead.
   *
   * @see https://www.drupal.org/node/3551152
   */
  public function getExpiry(): ?int;

  /**
   * Gets the restriction weight.
   *
   * @return int|null
   *   The weight.
   */
  public function getWeight(): ?int;

  /**
   * Gets the user restriction type plugin.
   *
   * @return \Drupal\user_restrictions\Plugin\UserRestrictionTypeInterface
   *   The user restriction type plugin.
   */
  public function getTypePlugin(): UserRestrictionTypeInterface;

  /**
   * Checks if the restriction matches the given data.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   *
   * @return bool
   *   TRUE when the restriction matches the data, FALSE otherwise.
   */
  public function matches(array $form, FormStateInterface $form_state, string $form_id): bool;

}
