<?php

declare(strict_types=1);

namespace Drupal\user_restrictions\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user_restrictions\Plugin\UserRestrictionTypeInterface;
use Drupal\user_restrictions\UserRestrictionTypePluginCollection;

/**
 * Defines a user restriction configuration entity.
 *
 * @ConfigEntityType(
 *   id = "user_restrictions",
 *   label = @Translation("User restrictions"),
 *   handlers = {
 *     "form" = {
 *       "add" = "Drupal\user_restrictions\Form\UserRestrictionsAddForm",
 *       "edit" = "Drupal\user_restrictions\Form\UserRestrictionsEditForm",
 *       "delete" = "Drupal\user_restrictions\Form\UserRestrictionsDeleteForm",
 *     },
 *     "list_builder" = "Drupal\user_restrictions\UserRestrictionsListBuilder",
 *   },
 *   admin_permission = "administer user restrictions",
 *   config_prefix = "user_restrictions",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "status" = "status",
 *     "weight" = "weight"
 *   },
 *   links = {
 *     "collection" = "/admin/config/people/user-restrictions",
 *     "delete-form" = "/admin/config/people/user-restrictions/manage/{user_restrictions}/delete",
 *     "disable" = "/admin/config/people/user-restrictions/manage/{user_restrictions}/disable",
 *     "edit-form" = "/admin/config/people/user-restrictions/manage/{user_restrictions}",
 *     "enable" = "/admin/config/people/user-restrictions/manage/{user_restrictions}/enable",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "weight",
 *     "plugin",
 *     "pattern",
 *     "pattern_type",
 *     "forms",
 *     "no_expiration",
 *     "expiration",
 *   }
 * )
 */
class UserRestrictions extends ConfigEntityBase implements UserRestrictionInterface {

  /**
   * The value for permanent restrictions.
   *
   * @deprecated in user_restrictions:2.1.1 and is removed from
   *   user_restriction:3.0.0. There is no replacement.
   *
   * @see https://www.drupal.org/node/3551042
   *
   * @noinspection PhpUnused
   */
  const int NO_EXPIRY = 2147483647;

  /**
   * The type for restrictions that reject values.
   *
   * @deprecated in user_restrictions:2.1.1 and is removed from
   *   user_restriction:3.0.0. There is no replacement.
   *
   * @see https://www.drupal.org/node/3551042
   *
   * @noinspection PhpUnused
   */
  const bool BLACKLIST = FALSE;

  /**
   * The type for restrictions that allow using values.
   *
   * @deprecated in user_restrictions:2.1.1 and is removed from
   *   user_restriction:3.0.0. Use
   *   \Drupal\user_restrictions\UserRestrictionInterface::PATTERN_ALLOW
   *   instead.
   *
   * @see https://www.drupal.org/node/3551042
   *
   * @noinspection PhpUnused
   */
  const bool WHITELIST = TRUE;

  /**
   * The ID.
   *
   * @var string
   */
  protected string $id = '';

  /**
   * The label.
   *
   * @var string
   */
  protected string $label = '';

  /**
   * The weight.
   *
   * @var int
   */
  protected int $weight = 0;

  /**
   * The user restriction type plugin ID.
   *
   * @var string
   */
  protected string $plugin = 'user_restrictions_username';

  /**
   * The user restriction type plugin collection.
   *
   * @var \Drupal\user_restrictions\UserRestrictionTypePluginCollection|null
   */
  protected ?UserRestrictionTypePluginCollection $typePluginCollection = NULL;

  /**
   * The pattern.
   *
   * @var string
   */
  protected string $pattern = '';

  /**
   * The pattern type, 0 when the pattern rejects values.
   *
   * @var int
   */
  protected int $pattern_type = 0;

  /**
   * The forms to which the restriction applies.
   *
   * @var array
   */
  protected array $forms = [];

  /**
   * Whether the restriction has an expiration time.
   *
   * @var bool
   */
  protected bool $no_expiration = TRUE;

  /**
   * The restriction expiration time.
   *
   * @var int
   */
  protected int $expiration = 0;

  /**
   * Allows to access deprecated/removed properties.
   *
   * @noinspection PhpMixedReturnTypeCanBeReducedInspection
   */
  public function __get(string $name): mixed {
    return match ($name) {
      'access_type' => (bool) ($this->pattern_type & UserRestrictionInterface::PATTERN_ALLOW),
      'expiry' => $this->expiration ?? 0,
      'rule_type' => $this->plugin ?? NULL,
      'name' => $this->id ?? NULL,
      default => NULL,
    };
  }

  /**
   * Allows to set deprecated/removed properties.
   */
  public function __set(string $name, mixed $value): void {
    switch ($name) {
      case 'access_type':
        @trigger_error('Accessing the access_type property is deprecated in user_restriction:2.1.1 and is removed from user_restrictions:3.0.0. Use the ' . __CLASS__ . '::pattern_type property instead. See https://www.drupal.org/node/3548381', E_USER_DEPRECATED);

        if ($value) {
          $this->pattern_type |= UserRestrictionInterface::PATTERN_ALLOW;
        }
        break;

      case 'expiry':
        @trigger_error('Accessing the expiry property is deprecated in user_restriction:2.1.1 and is removed from user_restrictions:3.0.0. Use the ' . __CLASS__ . '::expiration property instead. See https://www.drupal.org/node/3551152', E_USER_DEPRECATED);

        $this->expiration = $value;
        break;

      case 'name':
        @trigger_error('Accessing the name property is deprecated in user_restriction:2.1.1 and is removed from user_restrictions:3.0.0. Use the ' . __CLASS__ . '::id property instead. See https://www.drupal.org/node/3551072', E_USER_DEPRECATED);

        $this->id = $value;
        break;

      case 'rule_type':
        @trigger_error('Accessing the rule_type property is deprecated in user_restriction:2.1.1 and is removed from user_restrictions:3.0.0. Use the ' . __CLASS__ . '::plugin property instead. See https://www.drupal.org/node/3554615', E_USER_DEPRECATED);

        $this->plugin = $value;
        break;
    }
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpMixedReturnTypeCanBeReducedInspection
   */
  public function set($property_name, $value): mixed {
    if (in_array($property_name, ['expiration', 'expiry'])) {
      if ($value instanceof DrupalDateTime) {
        $this->expiration = $value->getTimestamp();
      }

      return $this;
    }
    else {
      return parent::set($property_name, $value);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getTypePluginId(): string {
    return $this->plugin;
  }

  /**
   * {@inheritdoc}
   */
  public function getRuleType(): string {
    @trigger_error('The ' . __METHOD__ . '() method is deprecated in user_restriction:2.1.1 and is removed from user_restrictions:3.0.0. Use the ' . __CLASS__ . '::getPluginId() method instead. See https://www.drupal.org/node/3554615', E_USER_DEPRECATED);

    return $this->plugin;
  }

  /**
   * {@inheritdoc}
   */
  public function getPattern(): string {
    return $this->pattern;
  }

  /**
   * {@inheritdoc}
   */
  public function getPatternType(): int {
    return $this->pattern_type;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccessType(): bool {
    @trigger_error('The ' . __METHOD__ . '() method is deprecated in user_restriction:2.1.1 and is removed from user_restrictions:3.0.0. Use the ' . __CLASS__ . '::getPatternType() method instead. See https://www.drupal.org/node/3548381', E_USER_DEPRECATED);

    return (bool) ($this->pattern_type & UserRestrictionInterface::PATTERN_ALLOW);
  }

  /**
   * {@inheritdoc}
   */
  public function getForms(): array {
    return $this->forms ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function expires(): bool {
    return !$this->no_expiration;
  }

  /**
   * {@inheritdoc}
   */
  public function getExpiration(): ?int {
    return $this->expiration ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getExpiry(): ?int {
    @trigger_error('The' . __METHOD__ . '() is deprecated in user_restriction:2.1.1 and is removed from user_restrictions:3.0.0. Use the ' . __CLASS__ . 'getExpiration() method instead. See https://www.drupal.org/node/3551152', E_USER_DEPRECATED);

    return $this->expiration ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): ?int {
    return $this->weight ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getTypePlugin(): UserRestrictionTypeInterface {
    return $this->getTypePluginCollection()->get($this->plugin);
  }

  /**
   * Gets the user restriction type plugin collection.
   *
   * @return \Drupal\user_restrictions\UserRestrictionTypePluginCollection
   *   The user restriction type plugin collection.
   *
   * @throws \InvalidArgumentException
   *   The plugin collection class has not been found.
   */
  protected function getTypePluginCollection(): UserRestrictionTypePluginCollection {
    if (!$this->typePluginCollection) {
      $this->typePluginCollection = \Drupal::classResolver(UserRestrictionTypePluginCollection::class)
        ->initializeCollection($this->plugin, $this->id());
    }

    return $this->typePluginCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function matches(array $form, FormStateInterface $form_state, string $form_id): bool {
    $forms = $this->getForms();

    // For backward compatibility, when the list of forms is empty, consider
    // the restriction to be applicable to all forms.
    if ($forms && !in_array($form_id, $forms, TRUE)) {
      return FALSE;
    }

    if ($data = $this->getTypePlugin()->buildData($form, $form_state, $form_id)) {
      return (bool) preg_match('/' . $this->pattern . '/i', $data);
    }

    return FALSE;
  }

}
