<?php

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpUnused */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUndefinedNamespaceInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\user_restrictions\Form;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\user_restrictions\Entity\UserRestrictionInterface;
use Drupal\user_restrictions\UserRestrictionTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the form to edit user restrictions.
 *
 * @internal
 */
class UserRestrictionsEditForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = new static($container->get('plugin.manager.user_restriction_type'));
    $instance->setStringTranslation($container->get('string_translation'));
    $instance->setLoggerFactory($container->get('logger.factory'));
    $instance->setMessenger($container->get('messenger'));

    return $instance;
  }

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Construct a new \Drupal\user_restrictions\Form\UserRestrictionsEditForm instance.
   *
   * @param \Drupal\user_restrictions\UserRestrictionTypeManagerInterface $typeManager
   *   The user restriction type manager.
   */
  public function __construct(
    protected UserRestrictionTypeManagerInterface $typeManager,
  ) {}

  // phpcs:enable

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    /** @var \Drupal\user_restrictions\UserRestrictionInterface $restriction */
    $restriction = $this->entity;
    $expiration = $restriction->getExpiration();

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#default_value' => $restriction->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#machine_name' => [
        'exists' => ['\Drupal\user_restrictions\Entity\UserRestrictions', 'load'],
      ],
      '#default_value' => $restriction->id(),
      '#disabled' => !$restriction->isNew(),
      '#required' => TRUE,
    ];

    $form['pattern_details'] = [
      '#type' => 'details',
      '#title' => $this->t('Pattern'),
      '#open' => TRUE,
    ];

    $form['pattern_details']['plugin'] = [
      '#type' => 'radios',
      '#title' => $this->t('Type'),
      '#default_value' => $restriction->getPluginId(),
      '#options' => $this->typeManager->getTypesAsOptions(),
      '#required' => TRUE,
    ];

    $regex_help = Url::fromUri('https://regex101.com');
    $redos = Url::fromUri('https://www.owasp.org/index.php/Regular_expression_Denial_of_Service_-_ReDoS');

    $form['pattern_details']['pattern'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pattern'),
      '#size' => 10,
      '#maxlength' => 64,
      '#default_value' => $restriction->getPattern(),
      '#field_prefix' => '/',
      '#field_suffix' => '/i',
      '#description' => $this->t(
        'Add a pattern for this rule to match.<br />Regular expressions are accepted and can be used for more complex restrictions. If you want to block patterns with regex characters in, you will need to escape them.<br /><a href=":regex">Test out regex patterns here first if you are unsure</a> and use appropriate caution when adding regex patterns to ensure you don\'t accidentally <a href=":redos">ReDos</a> yourself.',
        [':regex' => $regex_help->getUri(), ':redos' => $redos->getUri()]
      ),
      '#required' => TRUE,
    ];

    $form['pattern_details']['pattern_type'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Values matching the pattern are allowed'),
      '#return_value' => UserRestrictionInterface::PATTERN_ALLOW,
      '#default_value' => $restriction->getPatternType(),
    ];

    $form['pattern_details']['forms'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Forms'),
      '#default_value' => $restriction->getForms(),
      '#options' => [
        'user_form' => $this->t('User edit form'),
        'user_login_form' => $this->t('User login form'),
        'user_register_form' => $this->t('User registration form'),
      ],
      '#required' => TRUE,
    ];

    $form['expiration_details'] = [
      '#type' => 'details',
      '#title' => $this->t('Expiration'),
      '#description' => $this->t('Set a time for this restriction to expire or select <em>None</em> to create a permanent restriction.'),
      '#open' => TRUE,
    ];

    $form['expiration_details']['no_expiration'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('None'),
      '#default_value' => !$restriction->expires(),
    ];

    $form['expiration_details']['container'] = [
      '#type' => 'container',
      '#states' => [
        // Hide the datetime form element when the restriction does not expire.
        'invisible' => [
          'input[name="no_expiration"]' => ['checked' => TRUE],
        ],
      ],
    ];

    if ($expiration == 0 || $expiration === 2147483647) {
      // For compatibility with previous versions, if the expiration time has
      // not been set, set it to seven days after now.
      $default_expiration = new DrupalDateTime('now +7 days');
    }
    else {
      $default_expiration = DrupalDateTime::createFromTimestamp($expiration);
    }

    $form['expiration_details']['container']['expiration'] = [
      '#type' => 'datetime',
      '#default_value' => $default_expiration,
      '#title_display' => 'invisible',
      '#required' => TRUE,
    ];

    $form['weight'] = [
      '#type' => 'value',
      '#value' => $restriction->getWeight(),
    ];

    return parent::form($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *    The entity type does not exist.
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   *    The storage handler could not be loaded.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $query = $this->entityTypeManager->getStorage('user_restrictions')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('id', $form_state->getValue('id'), '<>')
      ->condition('plugin', $form_state->getValue('plugin'))
      ->condition('pattern', $form_state->getValue('pattern'));
    $existing = $query->execute();

    if (!empty($existing)) {
      $form_state->setError($form['pattern_details']['pattern'], $this->t('A rule with the same pattern already exists.'));
    }

    parent::validateForm($form, $form_state);
  }

}
