<?php

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpMissingParamTypeInspection */
/** @noinspection PhpUnused */
/** @noinspection PhpUnusedParameterInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUndefinedNamespaceInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\user_restrictions;

use Drupal\Core\Config\Entity\DraggableListBuilder;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user_restrictions\Entity\UserRestrictionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the list builder for user restriction entities.
 *
 * @see \Drupal\user_restrictions\Entity\UserRestriction
 *
 * @internal
 */
class UserRestrictionsListBuilder extends DraggableListBuilder {

  /**
   * {@inheritdoc}
   */
  protected $entitiesKey = 'user_restrictions';

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): self {
    $instance = new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('plugin.manager.user_restriction_type'),
    );
    $instance->setDateFormatter($container->get('date.formatter'));

    return $instance;
  }

  // phpcs:disable Drupal.Files.LineLength.TooLong, Drupal.Functions.MultiLineFunctionDeclaration.MissingTrailingComma

  /**
   * Constructs a new \Drupal\user_restrictions\UserRestrictionsListBuilder instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage.
   * @param \Drupal\user_restrictions\UserRestrictionTypeManagerInterface $typeManager
   *   The user restriction type manager.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    protected UserRestrictionTypeManagerInterface $typeManager,
  ) {
    parent::__construct($entity_type, $storage);
  }

  // phpcs:enable

  /**
   * Sets the date formatter for the list builder.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter to use.
   *
   * @return $this
   */
  protected function setDateFormatter(DateFormatterInterface $date_formatter): self {
    $this->dateFormatter = $date_formatter;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'user_restrictions_overview';
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['label'] = $this->t('Name');
    $header['plugin'] = $this->t('Type');
    $header['pattern'] = $this->t('Pattern');
    $header['pattern_type'] = $this->t('Pattern type');
    $header['expiration'] = $this->t('Expiration');
    $header['status'] = $this->t('Status');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\user_restrictions\Entity\UserRestrictionInterface $entity */
    $expiration = $entity->getExpiration();

    $row['label'] = $entity->label();
    $row['plugin']['data'] = [
      '#markup' => $entity->getTypePlugin()->getLabel(),
    ];
    $row['pattern']['data'] = [
      '#markup' => $entity->getPattern(),
    ];
    $row['pattern_type']['data'] = [
      '#markup' => ($entity->getPatternType() & UserRestrictionInterface::PATTERN_ALLOW) ? $this->t('Allow') : $this->t('Reject'),
    ];
    $row['expiration']['data'] = [
      '#markup' => $entity->expires() ? $this->dateFormatter->format($expiration, 'short') : $this->t('Never'),
    ];
    $row['status']['data'] = [
      '#markup' => $entity->status() ? $this->t('Enabled') : $this->t('Disabled'),
    ];

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function render(): array {
    $entities = $this->load();

    // If there are no multiple user restrictions, disable dragging by unsetting
    // the weight key.
    if (count($entities) <= 1) {
      unset($this->weightKey);
    }

    return parent::render();
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   *   The entity passed as parameter is malformed.
   * @throws \Drupal\Core\LinkTemplate\UndefinedLinkTemplateException
   *   The link template is undefined.
   */
  public function getDefaultOperations(EntityInterface $entity): array {
    $operations = parent::getDefaultOperations($entity);

    /** @var \Drupal\user_restrictions\UserRestrictionInterface $entity */
    if (!$entity->status()) {
      // Remove disable and edit operations for disabled restrictions.
      if (isset($operations['disable'])) {
        unset($operations['disable']);
      }
      if (isset($operations['edit'])) {
        unset($operations['edit']);
      }
    }

    return $operations;
  }

}
