<?php

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpMissingParamTypeInspection */
/** @noinspection PhpUnused */
/** @noinspection PhpUnusedParameterInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUndefinedNamespaceInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\user_restrictions;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user_restrictions\Entity\UserRestrictionInterface;
use Psr\Log\LoggerInterface;

/**
 * Defines the user restriction manager.
 *
 * @internal
 */
class UserRestrictionsManager implements UserRestrictionsManagerInterface {

  use StringTranslationTrait;

  /**
   * List of restriction errors.
   *
   * @var string[]
   */
  protected array $errors = [];

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $entityStorage;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * Constructs a UserRestrictionsManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_manager
   *   The entity storage.
   * @param \Psr\Log\LoggerInterface|\Drupal\user_restrictions\UserRestrictionTypeManagerInterface $logger
   *   The logger channel.
   * @param \Drupal\Component\Datetime\TimeInterface|\Psr\Log\LoggerInterface $time
   *   The time service.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *   An error happened when creating a plugin instance.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_manager,
    LoggerInterface|UserRestrictionTypeManagerInterface $logger,
    TimeInterface|LoggerInterface $time,
  ) {
    $this->entityStorage = $entity_manager->getStorage('user_restrictions');
    $this->logger = $logger;
    $this->time = $time;

    if ($logger instanceof UserRestrictionTypeManagerInterface) {
      @trigger_error('Calling ' . __CLASS__ . '::__construct() with the $type_manager argument is deprecated in user_restrictions:2.1.1 and removed in user_restrictions:3.0.0. See https://www.drupal.org/node/3555657', E_USER_DEPRECATED);

      $this->logger = $time;
      $this->time = func_get_arg(3);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function matchesRestrictions(array $form, FormStateInterface $form_state, string $form_id): bool {
    $query = $this->entityStorage->getQuery()->accessCheck(FALSE);

    // no_expiration = TRUE OR
    // ((no_expiration IS NULL OR no_expiration = FALSE) AND expiration > now)
    $no_expiration_group = $query->orConditionGroup()
      ->condition('no_expiration', FALSE)
      ->condition('no_expiration', NULL, 'IS NULL');
    $expiration_group = $query->andConditionGroup()
      ->condition($no_expiration_group)
      ->condition('expiration', $this->time->getRequestTime(), '>=');
    $or_group = $query->orConditionGroup()
      ->condition('no_expiration', TRUE)
      ->condition($expiration_group);

    $query->condition('status', TRUE)
      ->condition($or_group)
      ->sort('weight');
    $results = $query->execute();

    if (!empty($results)) {
      foreach ($this->entityStorage->loadMultiple($results) as $restriction) {
        /** @var \Drupal\user_restrictions\Entity\UserRestrictionInterface $restriction */
        if ($restriction->matches($form, $form_state, $form_id)) {
          if (!($restriction->getPatternType() & UserRestrictionInterface::PATTERN_ALLOW)) {
            $type_plugin = $restriction->getTypePlugin();
            $log_message = $type_plugin->getLogMessage();
            $log_message[1]['%restriction'] = $restriction->label();

            try {
              $log_message[1]['link'] = $restriction->toLink($this->t('Edit restriction'), 'edit-form')->toString();
            }
            catch (EntityMalformedException) {
              // Ignore the exception, which was thrown when logging a notice
              // message.
            }

            $this->logger->notice($log_message[0], $log_message[1]);
            $this->setError($restriction->getTypePluginId(), $type_plugin->getErrorMessage());
          }

          return TRUE;
        }
      }
    }

    // No restrictions match.
    return FALSE;
  }

  /**
   * Sets the error message for a specific restriction.
   *
   * @param string $type
   *   The restriction type (for example, name).
   * @param string $message
   *   Error message.
   *
   * @return $this
   */
  protected function setError($type, $message): self {
    $this->errors[$type] = $message;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getErrors(): array {
    return $this->errors;
  }

  /**
   * {@inheritdoc}
   */
  public function deleteExpiredRules(): self {
    $query = $this->entityStorage->getQuery()->accessCheck(FALSE);
    $or_group = $query->orConditionGroup()
      ->condition('no_expiration', FALSE)
      ->condition('no_expiration', NULL, 'IS NULL');
    $query->condition('status', TRUE)
      ->condition($or_group)
      ->condition('expiration', $this->time->getRequestTime(), '<=');
    $results = $query->execute();

    if (!empty($results)) {
      /** @var \Drupal\user_restrictions\Entity\UserRestrictionInterface $restriction */
      foreach ($this->entityStorage->loadMultiple($results) as $restriction) {
        try {
          $restriction->delete();
          $this->logger->notice('Expired rule %label has been deleted.', ['%label' => $restriction->label()]);
        }
        catch (EntityStorageException) {
          // Ignore the exception. Expired restrictions can be deleted the next
          // time cron tasks are run.
        }
      }
    }

    return $this;
  }

}
