<?php

/**
 * @file
 * Post update functions for the User Restrictions module.
 */

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpUnused */

// phpcs:enable

use Drupal\Core\Config\Entity\ConfigEntityUpdater;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\user\RoleInterface;
use Drupal\user_restrictions\Entity\UserRestrictionInterface;

/**
 * Initializes the new user restriction configuration properties.
 */
function user_restrictions_post_update_config_properties(?array &$sandbox = NULL): void {
  Drupal::classResolver(ConfigEntityUpdater::class)
    ->update($sandbox, 'user_restrictions', function (UserRestrictionInterface $restriction): bool {
      $needs_update = FALSE;

      if ($restriction->get('expiration') == NULL) {
        $needs_update = TRUE;

        if (($expiration = $restriction->get('expiry')) == 2147483647) {
          // 2147483647, the maximum value for a 32-bit integer in PHP, was
          // previously used for permanent restrictions.
          $expiration = 0;
        }

        $restriction->set('expiration', $expiration);
      }

      if ($restriction->get('id') == NULL) {
        $needs_update = TRUE;
        $restriction->set('id', $restriction->get('name'));
      }

      if ($restriction->get('pattern_type') == NULL) {
        $needs_update = TRUE;
        $restriction->set(
          'pattern_type',
          $restriction->get('access_type') ? UserRestrictionInterface::PATTERN_ALLOW : 0
        );
      }

      return $needs_update;
    });
}

/**
 * Updates the "bypass user restrictions" permission for all roles.
 */
function user_restrictions_post_update_permissions(?array &$sandbox = NULL): void {
  Drupal::classResolver(ConfigEntityUpdater::class)
    ->update($sandbox, 'user_role', function (RoleInterface $role): bool {
      $needs_update = FALSE;

      if ($role->hasPermission('bypass user restrictions rules')) {
        $needs_update = TRUE;
        $role->revokePermission('bypass user restrictions rules');
        $role->grantPermission('bypass user restrictions');
      }

      return $needs_update;
    });
}

/**
 * Initializes the "no expiration" property for all restrictions.
 */
function user_restrictions_post_update_no_expiration(?array &$sandbox = NULL): void {
  Drupal::classResolver(ConfigEntityUpdater::class)
    ->update($sandbox, 'user_restrictions', function (UserRestrictionInterface $restriction): bool {
      $needs_update = FALSE;
      $expiration = $restriction->get('expiration');

      if (is_int($expiration)) {
        $needs_update = TRUE;
        $no_expiration = ($expiration == 0 || $expiration == 2147483647);
        $restriction->set('no_expiration', $no_expiration);

        if ($no_expiration) {
          $restriction->set('expiration', (new DrupalDateTime('now +7 days'))->getTimestamp());
        }
      }
      elseif (is_null($expiration)) {
        $needs_update = TRUE;
        $restriction->set('no_expiration', TRUE);
        $restriction->set('expiration', (new DrupalDateTime('now +7 days'))->getTimestamp());
      }

      return $needs_update;
    });
}

/**
 * Initializes the weight property for all the restrictions.
 */
function user_restrictions_post_update_weight(?array &$sandbox = NULL): void {
  Drupal::classResolver(ConfigEntityUpdater::class)
    ->update($sandbox, 'user_restrictions', function (UserRestrictionInterface $restriction): bool {
      $needs_update = FALSE;

      if ($restriction->get('weight') === NULL) {
        $needs_update = TRUE;
        $restriction->set('weight', 0);
      }

      return $needs_update;
    });
}

/**
 * Initializes the plugin ID property for all the restrictions.
 */
function user_restrictions_post_update_plugin_id(?array &$sandbox = NULL): void {
  Drupal::classResolver(ConfigEntityUpdater::class)
    ->update($sandbox, 'user_restrictions', function (UserRestrictionInterface $restriction): bool {
      $needs_update = FALSE;

      if ($restriction->get('plugin') === NULL) {
        $needs_update = TRUE;
        $restriction->set('plugin', $restriction->get('rule_type'));
      }

      return $needs_update;
    });
}

/**
 * Changes the user restriction type plugin IDs used by restrictions.
 */
function user_restrictions_post_update_change_plugin_id(?array &$sandbox = NULL): void {
  Drupal::classResolver(ConfigEntityUpdater::class)
    ->update($sandbox, 'user_restrictions', function (UserRestrictionInterface $restriction): bool {
      $needs_update = FALSE;
      $new_plugin_ids = [
        'client_ip' => 'user_restrictions_client_ip',
        'mail' => 'user_restrictions_email',
        'name' => 'user_restrictions_username',
      ];
      $plugin = $restriction->get('plugin');

      if (isset($new_plugin_ids[$plugin])) {
        $needs_update = TRUE;
        $restriction->set('plugin', $new_plugin_ids[$plugin]);
      }

      return $needs_update;
    });
}
