<?php

namespace Drupal\user_restrictions\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user_restrictions\Form\RestrictionDisableForm;
use Drupal\user_restrictions\Form\RestrictionEnableForm;
use Drupal\user_restrictions\Form\UserRestrictionsAddForm;
use Drupal\user_restrictions\Form\UserRestrictionsEditForm;
use Drupal\user_restrictions\Form\UserRestrictionsDeleteForm;
use Drupal\user_restrictions\UserRestrictionInterface;
use Drupal\user_restrictions\UserRestrictionsListBuilder;

/**
 * Defines the user restriction configuration entity.
 */
#[ConfigEntityType(
  id: 'user_restriction',
  label: new TranslatableMarkup('User restriction'),
  config_prefix: 'user_restriction',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'status' => 'status',
  ],
  handlers: [
    'form' => [
      'add' => UserRestrictionsAddForm::class,
      'edit' => UserRestrictionsEditForm::class,
      'delete' => UserRestrictionsDeleteForm::class,
      'disable' => RestrictionDisableForm::class,
      'enable' => RestrictionEnableForm::class,
    ],
    'list_builder' => UserRestrictionsListBuilder::class,
  ],
  links: [
    'edit-form' => '/admin/config/people/user-restrictions/manage/{user_restriction}',
    'delete-form' => '/admin/config/people/user-restrictions/manage/{user_restriction}/delete',
    'disable' => '/admin/config/people/user-restrictions/manage/{user_restriction}/disable',
    'enable' => '/admin/config/people/user-restrictions/manage/{user_restriction}/enable',
    'collection' => '/admin/config/people/user-restrictions',
  ],
  admin_permission: 'administer user restrictions',
  label_count: [
    'singular' => '@count user restriction',
    'plural' => '@count user restrictions',
  ],
  config_export: [
    'id',
    'label',
    'rule_type',
    'pattern',
    'pattern_type',
    'forms',
    'no_expiration',
    'expiration',
  ]
)]
class UserRestriction extends ConfigEntityBase implements UserRestrictionInterface {

  /**
   * The ID.
   *
   * @var string
   */
  protected string $id = '';

  /**
   * The label.
   *
   * @var string
   */
  protected string $label = '';

  /**
   * The restriction type.
   *
   * @var string
   */
  protected string $rule_type = 'name';

  /**
   * The pattern.
   *
   * @var string
   */
  protected string $pattern = '';

  /**
   * The pattern type.
   *
   * @var int
   */
  protected int $pattern_type = UserRestrictionInterface::PATTERN_ALLOW;

  /**
   * The forms to which the restriction applies.
   *
   * @var array
   */
  protected array $forms = [];

  /**
   * The restriction expiration time.
   *
   * @var int
   */
  protected int $expiration = 0;

  /**
   * {@inheritdoc}
   */
  public function set($property_name, mixed $value): self {
    if ($property_name === 'expiration') {
      if ($value instanceof DrupalDateTime) {
        $this->expiration = $value->getTimestamp();
      }

      return $this;
    }
    else {
      return parent::set($property_name, $value);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getRuleType(): string {
    return $this->get('rule_type');
  }

  /**
   * {@inheritdoc}
   */
  public function getPattern(): string {
    return $this->get('pattern');
  }

  /**
   * {@inheritdoc}
   */
  public function getPatternType(): int {
    return $this->get('pattern_type');
  }

  /**
   * {@inheritdoc}
   */
  public function getForms(): array {
    return $this->get('forms');
  }

  /**
   * {@inheritdoc}
   */
  public function expires(): bool {
    return !$this->get('no_expiration');
  }

  /**
   * {@inheritdoc}
   */
  public function getExpiration(): ?int {
    return $this->get('expiration');
  }

}
