<?php

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpUndefinedNamespaceInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUnused */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\user_restrictions\Plugin\UserRestrictionType;

use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user_restrictions\Attribute\UserRestrictionType;
use Drupal\user_restrictions\UserRestrictionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Defines a restriction type for the client IP.
 */
#[UserRestrictionType(
  id: 'client_ip',
  label: new TranslatableMarkup('Client IP'),
  weight: 0,
)]
class ClientIp extends UserRestrictionTypeBase {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->requestStack = $container->get('request_stack');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function matches(array $data, string $form_id = ''): false|UserRestrictionInterface {
    $client_ip = $this->requestStack->getCurrentRequest()->getClientIp();

    /** @var \Drupal\user_restrictions\UserRestrictionInterface $restriction */
    if ($restriction = $this->matchesValue($client_ip, $form_id)) {
      try {
        $this->logger->notice(
          'Restricted client IP %client_ip matching %restriction has been blocked.', [
            '%client_ip' => $client_ip,
            '%restriction' => $restriction->label(),
            'link' => $restriction->toLink($this->t('Edit restriction'), 'edit-form'),
          ]);
      }
      catch (EntityMalformedException) {
        // Ignore the exception, which was thrown when logging a notice message.
      }
    }

    return $restriction;
  }

  /**
   * {@inheritdoc}
   */
  public function getErrorMessage(): TranslatableMarkup {
    return $this->t(
      'Accessing the site from the IP %value is not allowed.',
      ['%value' => $this->requestStack->getCurrentRequest()->getClientIp()]
    );
  }

}
