<?php

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpUndefinedNamespaceInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUnused */
/** @noinspection PhpUnusedParameterInspection */
/** @noinspection PhpFullyQualifiedNameUsageInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\user_restrictions\Hook;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;

/**
 * Form hook implementations for the User Restrictions module.
 */
class UserRestrictionsFormHooks {

  /**
   * The permission to bypass user restrictions.
   *
   * @var string
   */
  protected const string BYPASS_RESTRICTIONS = 'bypass user restrictions';

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter', order: Order::Last)]
  public function userFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    $validation_handlers = [
      'user_form' => 'validateEditedAccount',
      'user_login_form' => 'validateLoginAccount',
      'user_register_form' => 'validateRegisteredAccount',
    ];

    if (isset($validation_handlers[$form_id])) {
      $form['#validate'][] = [self::class, $validation_handlers[$form_id]];
    }
  }

  /**
   * Form validation handler for \Drupal\user\ProfileForm.
   */
  public static function validateEditedAccount(array &$form, FormStateInterface $form_state): void {
    // If the account is edited by somebody who has the permission to bypass
    // user restrictions, do not validate the edited account.
    if (\Drupal::currentUser()?->hasPermission(self::BYPASS_RESTRICTIONS)) {
      return;
    }

    static::validateAccount($form, $form_state, 'user_form');
  }

  /**
   * Form validation handler for \Drupal\user\Form\UserLoginForm.
   */
  public static function validateLoginAccount(array &$form, FormStateInterface $form_state): void {
    if (empty($uid = $form_state->get('uid'))) {
      // If UserLoginForm::validateAuthentication() has not stored in uid the
      // user ID, or the stored value is FALSE, the login attempt failed. In
      // this case, do not validate the account used to log in.
      return;
    }

    try {
      /** @var \Drupal\user\UserInterface $account */
      $account = \Drupal::entityTypeManager()
        ->getStorage('user')
        ->load($uid);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException) {
      // If it was not possible to load the account, do not check the
      // permissions it has.
      $account = NULL;
    }

    // If the account has the permission to bypass user restrictions, do not
    // validate the account.
    if ($account?->hasPermission(self::BYPASS_RESTRICTIONS)) {
      return;
    }

    static::validateAccount($form, $form_state, 'user_login_form');
  }

  /**
   * Form validation handler for \Drupal\user\RegisterForm.
   */
  public static function validateRegisteredAccount(array &$form, FormStateInterface $form_state): void {
    $admin = $form_state->getValue('administer_users');

    // If the account is registered by somebody who administers users and has
    // the permission to bypass user restrictions, do not validate the
    // registered account.
    if ($admin && \Drupal::currentUser()?->hasPermission(self::BYPASS_RESTRICTIONS)) {
      return;
    }

    static::validateAccount($form, $form_state, 'user_register_form');
  }

  /**
   * Checks whether the submitted values match any user restriction.
   *
   * A validation error will be shown for every restriction that matches the
   * submitted values.
   *
   * @param array $form
   *   The submitted form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  public static function validateAccount(array &$form, FormStateInterface $form_state, string $form_id = ''): void {
    /** @var Drupal\user_restrictions\UserRestrictionsManagerInterface $restriction_manager */
    $restriction_manager = \Drupal::service('user_restrictions.manager');

    if ($restriction_manager->matchesRestrictions($form_state->getValues(), $form_id)) {
      foreach ($restriction_manager->getErrors() as $message) {
        $form_state->setError($form, $message);
      }
    }
  }

}
