<?php

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpUndefinedNamespaceInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUnused */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\user_restrictions\Plugin\UserRestrictionType;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user_restrictions\UserRestrictionInterface;
use Drupal\user_restrictions\Plugin\UserRestrictionTypeInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for user restriction types.
 */
abstract class UserRestrictionTypeBase extends PluginBase implements UserRestrictionTypeInterface {

  use StringTranslationTrait;

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $entityStorage;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * The List of patterns for the restriction.
   *
   * @var string[]
   */
  protected array $patterns = [];

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\user_restrictions\Plugin\UserRestrictionType\UserRestrictionTypeBase instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_manager
   *   The entity storage.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger instance.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   *   The storage handler could not be loaded.
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   The entity type does not exist.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_manager,
    LoggerInterface $logger,
    TimeInterface $time,
    TranslationInterface $string_translation,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->entityStorage = $entity_manager->getStorage('user_restriction');
    $this->logger = $logger;
    $this->time = $time;
    $this->setStringTranslation($string_translation);
  }

  // phpcs:enable

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   *   The storage handler could not be loaded.
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   The entity type does not exist.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('logger.channel.user_restrictions'),
      $container->get('datetime.time'),
      $container->get('string_translation')
    );
  }

  /**
   * Checks if the specified value matches the restriction.
   *
   * @param string $value
   *   String to check against all restrictions of the type.
   * @param string $form_id
   *   The form to which the restriction is applied.
   *
   * @return bool|\Drupal\user_restrictions\UserRestrictionInterface
   *   The restriction entity if the value matches one of the restrictions,
   *   FALSE otherwise.
   */
  protected function matchesValue(string $value, string $form_id = ''): false|UserRestrictionInterface {
    $query = $this->entityStorage->getQuery()->accessCheck(FALSE);
    $or_group = $query->orConditionGroup()
      ->condition('no_expiration', TRUE)
      ->condition('expiration', $this->time->getRequestTime(), '>');
    $query->condition('rule_type', $this->getPluginId())
      ->condition('status', TRUE)
      ->condition($or_group);
    $return = FALSE;
    $results = $query->execute();

    if (empty($results)) {
      return FALSE;
    }

    /** @var \Drupal\user_restrictions\UserRestrictionInterface $restriction */
    foreach ($this->entityStorage->loadMultiple($results) as $restriction) {
      // For backward compatibility, when the list of the forms is empty,
      // consider the restriction to be applicable to all forms.
      if ($forms = $restriction->getForms()) {
        if ($form_id && !in_array($form_id, $forms)) {
          continue;
        }
      }

      if (preg_match('/' . $restriction->getPattern() . '/i', $value)) {
        if ($restriction->getPatternType() & UserRestrictionInterface::PATTERN_ALLOW) {
          return FALSE;
        }

        // If $value is disallowed, store the restriction but don't return, as
        // there may be a restriction allowing it further in the loop.
        $return = $restriction;
      }
    }

    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function getPatterns(): array {
    if (!empty($this->patterns)) {
      return $this->patterns;
    }

    $restrictions = $this->entityStorage
      ->loadByProperties(['rule_type' => $this->getPluginId()]);

    if (empty($restrictions)) {
      return [];
    }

    /** @var \Drupal\user_restrictions\UserRestrictionInterface $restriction */
    foreach ($restrictions as $id => $restriction) {
      $this->patterns[$id] = $restriction->getPattern();
    }

    return $this->patterns;
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): TranslatableMarkup {
    return $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getErrorMessage(): TranslatableMarkup {
    return $this->t('Using reserved data.');
  }

}
