<?php

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpUndefinedNamespaceInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUnused */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\user_restrictions;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class to build a listing of user restriction entities.
 *
 * @see \Drupal\user_restrictions\Entity\UserRestriction
 */
class UserRestrictionsListBuilder extends ConfigEntityListBuilder {

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): static {
    $instance = new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('user_restrictions.type_manager'),
    );
    $instance->setDateFormatter($container->get('date.formatter'));

    return $instance;
  }

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\user_restrictions\UserRestrictionsListBuilder instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage.
   * @param \Drupal\user_restrictions\UserRestrictionTypeManagerInterface $typeManager
   *   The user restriction type manager.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    protected UserRestrictionTypeManagerInterface $typeManager,
  ) {
    parent::__construct($entity_type, $storage);
  }

  // phpcs:enable

  /**
   * Sets the date formatter for the list builder.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter to use.
   *
   * @return $this
   */
  protected function setDateFormatter(DateFormatterInterface $date_formatter): self {
    $this->dateFormatter = $date_formatter;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['label'] = $this->t('Name');
    $header['rule_type'] = $this->t('Type');
    $header['pattern'] = $this->t('Pattern');
    $header['pattern_type'] = $this->t('Pattern type');
    $header['expiration'] = $this->t('Expiration');
    $header['status'] = $this->t('Status');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\user_restrictions\UserRestrictionInterface $entity */
    $expiration = $entity->getExpiration();

    $row['label'] = $entity->label();
    $row['rule_type'] = $this->typeManager->getType($entity->getRuleType())->getLabel();
    $row['pattern'] = $entity->getPattern();
    $row['pattern_type'] = ($entity->getPatternType() & UserRestrictionInterface::PATTERN_ALLOW) ? $this->t('Allow') : $this->t('Reject');
    $row['expiration'] = $entity->expires() ? $this->dateFormatter->format($expiration, 'short') : $this->t('Never');
    $row['status'] = $entity->status() ? $this->t('Enabled') : $this->t('Disabled');

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function render(): array {
    $build = parent::render();
    $build['table']['#empty'] = $this->t('There are currently no user restrictions. <a href=":url">Add a new one</a>.', [
      ':url' => Url::fromRoute('user_restrictions.add')->toString(),
    ]);

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity): array {
    $operations = parent::getDefaultOperations($entity);

    /** @var \Drupal\user_restrictions\UserRestrictionInterface $entity */
    if (!$entity->status()) {
      // Remove disable and edit operations for disabled restrictions.
      if (isset($operations['disable'])) {
        unset($operations['disable']);
      }
      if (isset($operations['edit'])) {
        unset($operations['edit']);
      }
    }

    return $operations;
  }

}
