<?php

// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
/** @noinspection PhpUndefinedNamespaceInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUnused */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\user_restrictions;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Psr\Log\LoggerInterface;

/**
 * The user restriction manager.
 */
class UserRestrictionsManager implements UserRestrictionsManagerInterface {

  use StringTranslationTrait;

  /**
   * List of restriction errors.
   *
   * @var string[]
   */
  protected array $errors = [];

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $entityStorage;

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\user_restrictions\UserRestrictionsManager instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_manager
   *   The entity storage.
   * @param \Drupal\user_restrictions\UserRestrictionTypeManagerInterface $typeManager
   *   The user restriction type manager.
   * @param Psr\Log\LoggerInterface $logger
   *   The logger channel.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *   An error happened when creating a plugin instance.
   *
   * @noinspection PhpDocSignatureInspection
   */
  public function __construct(
    EntityTypeManagerInterface $entity_manager,
    protected UserRestrictionTypeManagerInterface $typeManager,
    protected LoggerInterface $logger,
    protected TimeInterface $time,
  ) {
    $this->entityStorage = $entity_manager->getStorage('user_restriction');
  }

  //phpcs:enable

  /**
   * {@inheritdoc}
   */
  public function matchesRestrictions(array $data, string $form_id = ''): bool {
    foreach ($this->typeManager->getTypes() as $key => $type) {
      if ($type->matches($data, $form_id)) {
        $this->setError($key, $type->getErrorMessage());
        // Break after first match.
        return TRUE;
      }
    }

    // No restrictions match.
    return FALSE;
  }

  /**
   * Sets the error message for a specific restriction.
   *
   * @param string $type
   *   The restriction type (for example, name).
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $message
   *   The error message.
   *
   * @return $this
   */
  protected function setError(string $type, TranslatableMarkup $message): static {
    $this->errors[$type] = $message;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getErrors(): array {
    return $this->errors;
  }

  /**
   * {@inheritdoc}
   */
  public function deleteExpiredRules(): static {
    // Delete enabled restrictions that are expired.
    $query = $this->entityStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('status', TRUE)
      ->condition('no_expiration', FALSE)
      ->condition('expiration', $this->time->getRequestTime(), '<=');
    $results = $query->execute();

    if (!empty($results)) {
      /** @var \Drupal\user_restrictions\UserRestrictionInterface $restriction */
      foreach ($this->entityStorage->loadMultiple($results) as $restriction) {
        try {
          $restriction->delete();
          $this->logger->notice(
            'Expired rule %label has been deleted.',
            ['%label' => $restriction->label()]
          );
        }
        catch (EntityStorageException) {
          // Ignore the exception. Expired restrictions can be deleted the next
          // time cron tasks are run.
        }
      }
    }

    return $this;
  }

}
