<?php

namespace Drupal\user_status_switch\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for toggling user status.
 */
class UserStatusSwitchController extends ControllerBase
{

    /**
     * Toggles the active status of a user.
     *
     * @param \Drupal\user\UserInterface $user
     *   The user entity.
     * @param \Symfony\Component\HttpFoundation\Request $request
     *   The request object.
     *
     * @return \Symfony\Component\HttpFoundation\JsonResponse
     *   JSON response with the new status.
     */
    public function toggle(UserInterface $user, Request $request)
    {
        // Check if user has permission to administer users.
        if (!$this->currentUser()->hasPermission('administer users')) {
            return new JsonResponse([
                'success' => FALSE,
                'message' => $this->t('You do not have permission to administer users.'),
            ], 403);
        }

        // Manual CSRF Token Validation.
        $token = $request->query->get('token');
        $path = '/admin/people/toggle-status/' . $user->id();

        if (!$token || !\Drupal::csrfToken()->validate($token, $path)) {
            return new JsonResponse([
                'success' => FALSE,
                'message' => $this->t('Invalid security token. Please refresh the page and try again.'),
            ], 403);
        }

        // Toggle the status.
        $new_status = !$user->isActive();

        if ($new_status) {
            $user->activate();
        } else {
            $user->block();
        }

        try {
            // Save the user.
            $user->save();

            // Clear the entity cache.
            \Drupal::entityTypeManager()->getStorage('user')->resetCache([$user->id()]);

            return new JsonResponse([
                'success' => TRUE,
                'status' => (bool) $new_status,
                'message' => $new_status
                    ? $this->t('User activated successfully.')
                    : $this->t('User blocked successfully.'),
            ]);
        } catch (\Exception $e) {
            return new JsonResponse([
                'success' => FALSE,
                'message' => $this->t('An error occurred while updating the user.'),
            ], 500);
        }
    }

}
