<?php

namespace Drupal\user_status_switch\Controller;

use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for toggling user status.
 */
class UserStatusSwitchController extends ControllerBase
{

    /**
     * The CSRF token generator.
     *
     * @var \Drupal\Core\Access\CsrfTokenGenerator
     */
    protected $csrfToken;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * Constructs a UserStatusSwitchController object.
     *
     * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
     *   The CSRF token generator.
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager.
     */
    public function __construct(CsrfTokenGenerator $csrf_token, EntityTypeManagerInterface $entity_type_manager)
    {
        $this->csrfToken = $csrf_token;
        $this->entityTypeManager = $entity_type_manager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('csrf_token'),
            $container->get('entity_type.manager')
        );
    }

    /**
     * Toggles the active status of a user.
     *
     * @param \Drupal\user\UserInterface $user
     *   The user entity.
     * @param \Symfony\Component\HttpFoundation\Request $request
     *   The request object.
     *
     * @return \Symfony\Component\HttpFoundation\JsonResponse
     *   JSON response with the new status.
     */
    public function toggle(UserInterface $user, Request $request)
    {
        // Check if user has permission to administer users.
        if (!$this->currentUser()->hasPermission('administer users')) {
            return new JsonResponse([
                'success' => FALSE,
                'message' => $this->t('You do not have permission to administer users.'),
            ], 403);
        }

        // Manual CSRF Token Validation.
        $token = $request->query->get('token');
        $path = '/admin/people/toggle-status/' . $user->id();

        if (!$token || !$this->csrfToken->validate($token, $path)) {
            return new JsonResponse([
                'success' => FALSE,
                'message' => $this->t('Invalid security token. Please refresh the page and try again.'),
            ], 403);
        }

        // Toggle the status.
        $new_status = !$user->isActive();

        if ($new_status) {
            $user->activate();
        } else {
            $user->block();
        }

        try {
            // Save the user.
            $user->save();

            // Clear the entity cache.
            $this->entityTypeManager->getStorage('user')->resetCache([$user->id()]);

            return new JsonResponse([
                'success' => TRUE,
                'status' => (bool) $new_status,
                'message' => $new_status
                    ? $this->t('User activated successfully.')
                    : $this->t('User blocked successfully.'),
            ]);
        } catch (\Exception $e) {
            return new JsonResponse([
                'success' => FALSE,
                'message' => $this->t('An error occurred while updating the user.'),
            ], 500);
        }
    }

}
