<?php

namespace Drupal\user_status_switch\Plugin\views\field;

use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler to display quick user status switch.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("user_status_switch_field")
 */
class UserStatusSwitchField extends FieldPluginBase
{

    /**
     * The CSRF token generator.
     *
     * @var \Drupal\Core\Access\CsrfTokenGenerator
     */
    protected $csrfToken;

    /**
     * Constructs a UserStatusSwitchField object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
     *   The CSRF token generator.
     */
    public function __construct(array $configuration, $plugin_id, $plugin_definition, CsrfTokenGenerator $csrf_token)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->csrfToken = $csrf_token;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('csrf_token')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // Leave empty to avoid a query on this field.
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $user = $this->getEntity($values);

        if (!$user || !$this->view->getUser()->hasPermission('administer users')) {
            return [];
        }

        // Generate CSRF token for the specific route.
        $path = '/admin/people/toggle-status/' . $user->id();
        $token = $this->csrfToken->get($path);

        $build = [
            '#theme' => 'user_status_switch',
            '#user_id' => $user->id(),
            '#status' => $user->isActive(),
            '#csrf_token' => $token,
        ];

        $build['#attached']['library'][] = 'user_status_switch/toggle';

        return $build;
    }

}
