<?php

namespace Drupal\username_suggester\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Utility\Token;
use Egulias\EmailValidator\EmailValidator;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Returns responses for Username Suggester routes.
 */
class UsernameSuggestionController extends ControllerBase {

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The email validator.
   *
   * @var \Drupal\Component\Utility\EmailValidatorInterface
   */
  protected $emailValidator;

  /**
   * Constructs a UsernameSuggestionController object.
   *
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Component\Utility\EmailValidatorInterface $email_validator
   *   The email validator.
   */
  public function __construct(Token $token, ConfigFactoryInterface $config_factory, EmailValidator $email_validator) {
    $this->token = $token;
    $this->configFactory = $config_factory;
    $this->emailValidator = $email_validator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('token'),
      $container->get('config.factory'),
      $container->get('email.validator')
    );
  }

  /**
   * Generates a username suggestion.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response with the suggested username.
   */
  public function generate(Request $request) {
    $email = $request->query->get('mail');
    $usernameSuggestion = '';

    // Mail is not valid, return empty string:
    if (empty($email) || !$this->emailValidator->isValid($email)) {
      return new JsonResponse(['value' => '']);
    }
    $config = $this->configFactory->get('username_suggester.settings');
    $pattern = $config->get('generation_pattern') ?: '[username_suggester:mail_local]';

    // Replace tokens.
    $usernameSuggestion = $this->token->replacePlain($pattern, ['mail' => $email]);

    // Sanitize the username suggestion:
    $sanitizedUsernameSuggestion = $this->sanitizeUsername($usernameSuggestion);

    return new JsonResponse(['value' => $sanitizedUsernameSuggestion]);
  }

  /**
   * Sanitizes a username to ensure it's valid for Drupal.
   *
   * @param string $username
   *   The username to sanitize.
   *
   * @return string
   *   The sanitized username.
   */
  protected function sanitizeUsername($username) {
    // Remove any characters that are not alphanumeric, underscore, hyphen,
    // period, or @.
    // Drupal allows these characters in usernames by default.
    $username = preg_replace('/[^a-zA-Z0-9_\-\.@]/', '', $username);

    // Trim whitespace.
    $username = trim($username);

    return $username;
  }

}
