<?php

namespace Drupal\username_suggester\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\RendererInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Username Suggester settings for this site.
 */
class UsernameSuggesterSettingsForm extends ConfigFormBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a UsernameSuggesterSettingsForm object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(ModuleHandlerInterface $moduleHandler, RendererInterface $renderer) {
    $this->moduleHandler = $moduleHandler;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'username_suggester_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['username_suggester.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('username_suggester.settings');

    $tokenModuleInstalled = $this->moduleHandler->moduleExists('token');
    if ($tokenModuleInstalled) {
      $tokenTree = [
        '#theme' => 'token_tree_link',
        '#global_types' => TRUE,
        '#click_insert' => TRUE,
        '#show_restricted' => FALSE,
        '#recursion_limit' => 3,
      ];
      $renderedTokenTree = $this->renderer->render($tokenTree);
    }
    else {
      $renderedTokenTree = $this->t('Contrib token module is not installed. Please install the <a href="https://www.drupal.org/project/token" target="_blank">Token</a> module to use the token browser.');
    }

    $form['generation_pattern'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Generation Pattern'),
      '#description' => $this->t('Define how the username matches. Supports tokens. @browse_tokens_link', [
        '@browse_tokens_link' => $renderedTokenTree,
      ]),
      '#default_value' => $config->get('generation_pattern'),
      '#required' => TRUE,
    ];

    $form['trigger_event'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Trigger Event'),
      '#description' => $this->t("Enter the event name (e.g., 'blur', 'change', 'keyup delay:500ms')."),
      '#default_value' => $config->get('trigger_event'),
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('username_suggester.settings')
      ->set('generation_pattern', $form_state->getValue('generation_pattern'))
      ->set('trigger_event', $form_state->getValue('trigger_event'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}

