<?php

declare(strict_types=1);

namespace Drupal\Tests\username_suggester\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\user\UserInterface;

/**
 * Tests the Username Suggester module functionality via JavaScript.
 *
 * @group username_suggester
 */
class UsernameSuggesterTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'username_suggester',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set user registration to allow visitors to register.
    $this->config('user.settings')
      ->set('register', UserInterface::REGISTER_VISITORS)
      ->save();
  }

  /**
   * Tests username suggestion with default pattern (mail_local).
   */
  public function testUsernameSuggestionDefaultPattern(): void {
    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();

    // Fill in the email field.
    $mailField = $page->findField('mail');
    $this->assertNotNull($mailField);
    $mailField->setValue('testuser@example.com');

    // Trigger blur event to activate HTMX.
    $mailField->blur();

    // Wait for the username field to be populated.
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "testuser"', 10000);

    // Verify the username field has the expected value.
    $usernameField = $page->findField('name');
    $this->assertEquals('testuser', $usernameField->getValue());
  }

  /**
   * Tests username suggestion with full email pattern.
   */
  public function testUsernameSuggestionWithMailPattern(): void {
    // Set the pattern to use full email.
    $this->config('username_suggester.settings')
      ->set('generation_pattern', '[username_suggester:mail]')
      ->save();

    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();
    $mailField = $page->findField('mail');
    $mailField->setValue('john@example.com');
    $mailField->blur();

    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "john@example.com"', 10000);

    $usernameField = $page->findField('name');
    $this->assertEquals('john@example.com', $usernameField->getValue());
  }

  /**
   * Tests username suggestion with mail_domain pattern.
   */
  public function testUsernameSuggestionWithMailDomainPattern(): void {
    // Set the pattern to use email domain.
    $this->config('username_suggester.settings')
      ->set('generation_pattern', '[username_suggester:mail_domain]')
      ->save();

    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();
    $mailField = $page->findField('mail');
    $mailField->setValue('user@mydomain.org');
    $mailField->blur();

    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "mydomain.org"', 10000);

    $usernameField = $page->findField('name');
    $this->assertEquals('mydomain.org', $usernameField->getValue());
  }

  /**
   * Tests username suggestion with custom combined pattern.
   */
  public function testUsernameSuggestionWithCombinedPattern(): void {
    // Set a custom pattern with prefix.
    $this->config('username_suggester.settings')
      ->set('generation_pattern', 'user_[username_suggester:mail_local]')
      ->save();

    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();
    $mailField = $page->findField('mail');
    $mailField->setValue('testperson@example.com');
    $mailField->blur();

    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "user_testperson"', 10000);

    $usernameField = $page->findField('name');
    $this->assertEquals('user_testperson', $usernameField->getValue());
  }

  /**
   * Tests username suggestion sanitizes special characters.
   */
  public function testUsernameSuggestionSanitization(): void {
    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();
    $mailField = $page->findField('mail');

    // Email with + sign should have it removed.
    $mailField->setValue('test+filter@example.com');
    $mailField->blur();

    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "testfilter"', 10000);

    $usernameField = $page->findField('name');
    $this->assertEquals('testfilter', $usernameField->getValue());
  }

  /**
   * Tests that invalid email returns empty username when validation enabled.
   */
  public function testUsernameSuggestionWithEmailValidationEnabled(): void {
    // Ensure email validation is enabled (default).
    $this->config('username_suggester.settings')
      ->set('validate_email', TRUE)
      ->save();

    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();
    $mailField = $page->findField('mail');

    // Valid email should populate username.
    $mailField->setValue('validuser@example.com');
    $mailField->blur();

    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "validuser"', 10000);

    $usernameField = $page->findField('name');
    $this->assertEquals('validuser', $usernameField->getValue());

    // Clear username field for next test.
    $usernameField->setValue('');

    // Invalid email should not populate username.
    $mailField->setValue('invalid-email');
    $mailField->blur();

    $this->assertSession()->assertWaitOnAjaxRequest();
    // Wait a bit and verify username field is still empty.
    $this->assertJsCondition('document.getElementById("edit-name").value === ""', 5000);
    $this->assertEquals('', $usernameField->getValue());
  }

  /**
   * Tests that invalid email returns value when validation is disabled.
   */
  public function testUsernameSuggestionWithEmailValidationDisabled(): void {
    // Disable email validation.
    $this->config('username_suggester.settings')
      ->set('validate_email', FALSE)
      ->save();

    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();
    $mailField = $page->findField('mail');

    // Valid email should populate username.
    $mailField->setValue('validuser@example.com');
    $mailField->blur();

    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "validuser"', 10000);

    $usernameField = $page->findField('name');
    $this->assertEquals('validuser', $usernameField->getValue());

    // Invalid email should still generate a username when validation disabled.
    $mailField->setValue('invalid-email');
    $mailField->blur();

    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "invalid-email"', 10000);
    $this->assertEquals('invalid-email', $usernameField->getValue());
  }

  /**
   * Tests username suggestion with change trigger event.
   */
  public function testUsernameSuggestionWithChangeTrigger(): void {
    // Set trigger event to change.
    $this->config('username_suggester.settings')
      ->set('trigger_event', 'change')
      ->save();

    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();
    $mailField = $page->findField('mail');
    $mailField->setValue('changetest@example.com');

    // Trigger change event by clicking elsewhere.
    $page->findField('name')->click();

    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "changetest"', 10000);

    $usernameField = $page->findField('name');
    $this->assertEquals('changetest', $usernameField->getValue());
  }

  /**
   * Tests username suggestion with keyup trigger event.
   */
  public function testUsernameSuggestionWithKeyupTrigger(): void {
    // Set trigger event to keyup with delay.
    $this->config('username_suggester.settings')
      ->set('trigger_event', 'keyup delay:500ms')
      ->save();

    $this->drupalGet('user/register');

    $page = $this->getSession()->getPage();
    $mailField = $page->findField('mail');

    // Type email character by character to trigger keyup events.
    $mailField->setValue('keyuptest@example.com');

    // Wait for the delay and HTMX request to complete.
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->assertJsCondition('document.getElementById("edit-name").value === "keyuptest"', 10000);

    $usernameField = $page->findField('name');
    $this->assertEquals('keyuptest', $usernameField->getValue());
  }

}
