<?php

namespace Drupal\username_suggestions\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class initialization.
 */
class UsernameSuggestionsController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * The Configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $request;

  /**
   * Constructs a UsernameSuggestionsController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request
   *   The Request object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, RequestStack $request) {
    $this->configFactory = $config_factory;
    $this->request = $request->getCurrentRequest();
  }

  /**
   * Generates username suggestions based on the given values.
   */
  public function getSuggestions(array $values) {
    $config = $this->config('username_suggestions.settings');
    $max_allowed = $config->get('username_suggestions_max');
    $suggestions_rules = $config->get('username_suggestions_rules') ?? "[username][random-xx]\n[email-firstpart][random-xx]";

    $username = $values['name'] ?? '';
    $emailFirstPart = strstr($values['mail'] ?? '', '@', TRUE);
    $replacements = [
      '[username]' => $username,
      '[email-firstpart]' => $emailFirstPart,
      '[random-x]' => rand(1, 9),
      '[random-xx]' => rand(10, 99),
      '[random-xxx]' => rand(100, 999),
      '[random-xxxx]' => rand(1000, 9999),
    ];

    // Generate initial suggestions using the configured rules.
    $suggestions = [];
    $rules = explode("\n", $suggestions_rules);
    foreach ($rules as $rule) {
      $suggestion = str_replace(array_keys($replacements), $replacements, trim($rule));
      if (!empty($suggestion)) {
        $suggestions[] = $suggestion;
      }
    }

    // Generate additional suggestions based on username variations.
    $suggestions[] = $username . rand(10, 99);
    $suggestions[] = $username . '_' . rand(100, 999);
    $suggestions[] = $username . rand(1, 9);
    $suggestions[] = $username . '_' . $emailFirstPart;

    // Ensure suggestions start with the same characters as the username.
    foreach ($suggestions as &$suggestion) {
      if (strpos($suggestion, $username) !== 0) {
        $suggestion = $username . $suggestion;
      }
    }

    // Filter suggestions to ensure they are unique and valid.
    $suggestions = array_unique($suggestions);

    // Limit the number of suggestions to the max allowed.
    $suggestions = array_slice($suggestions, 0, $max_allowed);

    return $suggestions;
  }

  /**
   * Handles AJAX requests for username suggestions.
   */
  public function ajaxSuggestions() {
    $values = $this->request->query->all();
    $suggestions = $this->getSuggestions($values);
    return new JsonResponse($suggestions);
  }

}
