<?php

namespace Drupal\users_list\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Controller for users list pages.
 */
class UsersListController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The default number of users to list.
   *
   * @var int
   */
  protected $defaultNumberOfUsers;

  /**
   * Constructs a UsersListController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory) {
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->defaultNumberOfUsers = $this->configFactory->get('users_list.settings')->get('users_list_default_number') ?? 10;
  }

  /**
   * Render users list based on type.
   */

  protected function buildList($result, $header, $header_desc, $empty_msg) {
    $users = [];
    foreach ($result as $record) {
      $users[] = [
        'uid' => $record->uid,
        'name' => $record->name,
        'roles' => $record->roles_target_id ?? [],
      ];
    }

    $links = [
      'all' => [
        'title' => $this->t('All Users'),
        'url' => Url::fromRoute('users_list.all'),
      ],
      'alpha' => [
        'title' => $this->t('Users A–Z'),
        'url' => Url::fromRoute('users_list.alpha'),
      ],
      'newest' => [
        'title' => $this->t('Newest Users'),
        'url' => Url::fromRoute('users_list.newest'),
      ],
      'roles' => [
        'title' => $this->t('Users by Roles'),
        'url' => Url::fromRoute('users_list.roles'),
      ],
      'content' => [
        'title' => $this->t('Users by Content'),
        'url' => Url::fromRoute('users_list.content'),
      ],
    ];

    return [
      '#theme' => 'users_list',
      '#header' => $header,
      '#header_desc' => $header_desc,

      '#navigation' => [
        '#type' => 'links',
        '#links' => $links,
        '#attributes' => ['class' => ['users-list-navigation']],
      ],

      '#users' => $users,
      '#empty_msg' => empty($users) ? $empty_msg : '',
      '#cache' => [
        'tags' => ['config:users_list.settings'],
        'contexts' => ['user.roles'],
      ],
    ];
  }

  public function allUsers() {
    return $this->buildList(
      $this->getAllUsers(),
      $this->t('All Users'),
      '',
      $this->t('There are currently no users.')
    );
  }

  public function alphaUsers() {
    return $this->buildList(
      $this->getUsersByAlpha(),
      $this->t('Users A–Z'),
      $this->t('Users listed alphabetically.'),
      $this->t('No users found.')
    );
  }

  public function newestUsers() {
    return $this->buildList(
      $this->getNewestUsers(),
      $this->t('Newest Users'),
      $this->t('List of newest registered users.'),
      $this->t('No users found.')
    );
  }

  public function usersByRoles() {
    $roles = array_keys(
      $this->entityTypeManager->getStorage('user_role')->loadMultiple()
    );

    return $this->buildList(
      $this->getUsersByRoles($roles),
      $this->t('Users by Roles'),
      $this->t('Users with assigned roles.'),
      $this->t('No users found.')
    );
  }

  public function usersByContent() {
    $types = array_keys(
      $this->entityTypeManager->getStorage('node_type')->loadMultiple()
    );

    return $this->buildList(
      $this->getUsersByContent($types),
      $this->t('Users by Content'),
      $this->t('Users who created content.'),
      $this->t('No users found.')
    );
  }
  /**
   * Query to get users alphabetically.
   */
  protected function getUsersByAlpha() {
    $query = $this->database->select('users_field_data', 'u')
      ->fields('u', ['uid', 'name'])
      ->condition('u.status', 1)
      ->orderBy('u.name', 'ASC')
      ->range(0, $this->defaultNumberOfUsers);
    return $query->execute()->fetchAll();
  }

  /**
   * Query to get users by content type.
   */
  protected function getUsersByContent($types) {
    $query = $this->database->select('users_field_data', 'u');
    $query->distinct();
    $query->innerJoin('node_field_data', 'n', 'n.uid = u.uid');
    $query->fields('u', ['uid', 'name']);
    $query->condition('u.status', 1);

    // Use 'IN' operator when passing an array of types.
    $query->condition('n.type', $types, 'IN');

    $query->orderBy('u.name', 'ASC')
      ->range(0, $this->defaultNumberOfUsers);

    return $query->execute()->fetchAll();
  }

  /**
   * Query to get users by role.
   */
  protected function getUsersByRoles(array $role_type) {
    $query = $this->database->select('users_field_data', 'u');
    $query->distinct();
    $query->fields('u', ['uid', 'name']);
    $query->condition('u.status', 1);
    $query->innerJoin('user__roles', 'r', 'r.entity_id = u.uid');
    // Use 'IN' operator for array of roles.
    $query->fields('r', ['roles_target_id']);
    $query->condition('r.roles_target_id', $role_type, 'IN');
    $query->orderBy('u.name', 'ASC')
      ->range(0, $this->defaultNumberOfUsers);
    return $query->execute()->fetchAll();
  }

  /**
   * Query to get newest users.
   */
  protected function getNewestUsers() {
    $query = $this->database->select('users_field_data', 'u')
      ->fields('u', ['uid', 'name'])
      ->condition('u.status', 1)
      ->orderBy('u.created', 'DESC')
      ->range(0, $this->defaultNumberOfUsers);
    return $query->execute()->fetchAll();
  }

  /**
   * Query to get users not in A-Z.
   */
  protected function getUsersNotInAlpha() {
    $query = $this->database->select('users_field_data', 'u')
      ->fields('u', ['uid', 'name'])
      ->condition('u.status', 1)
      ->where('LOWER(u.name) < :a OR LOWER(u.name) > :z', [':a' => 'a', ':z' => 'z'])
      ->orderBy('u.name', 'ASC')
      ->range(0, $this->defaultNumberOfUsers);
    return $query->execute()->fetchAll();
  }

  /**
   * Query to get all users.
   */
  protected function getAllUsers() {
    $query = $this->database->select('users_field_data', 'u')
      ->fields('u', ['uid', 'name'])
      ->condition('u.status', 1)
      ->orderBy('u.name', 'ASC')
      ->range(0, $this->defaultNumberOfUsers);
    return $query->execute()->fetchAll();
  }

}
