<?php

namespace Drupal\uswds_blb_configuration\Plugin\UswdsStyles\StylesGroup;

use Drupal\Core\Form\FormStateInterface;
use Drupal\uswds_blb_configuration\ResponsiveTrait;
use Drupal\uswds_blb_configuration\StylesGroup\StylesGroupPluginBase;

/**
 * Spacing styles group.
 *
 * @package Drupal\uswds_blb_configuration\Plugin\StylesGroup
 *
 * @StylesGroup(
 *   id = "spacing",
 *   title = @Translation("Spacing"),
 *   weight = 3,
 *   icon = "uswds_blb_configuration/images/plugins/spacing-icon.svg"
 * )
 */
class Spacing extends StylesGroupPluginBase {

  use ResponsiveTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['spacing'] = [
      '#type' => 'details',
      '#title' => $this->t('Spacing'),
      '#open' => FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildStyleFormElements(array &$form, FormStateInterface $form_state, $storage) {
    $this->buildBreakpointsFields($form, 'spacing');

    $form['spacing_preview'] = [
      '#theme' => 'spacing_preview',
    ];

    return $form;
  }

  /**
   * Helper function to get the options of given style name.
   *
   * @param string $name
   *   A config style name like background_color.
   *
   * @return array
   *   Array of key => value of style name options.
   */
  public function getStyleOptions(string $name) {
    $config = $this->configFactory->get('uswds_blb_configuration.settings');
    $config_options = $config->get($name);

    $options = ['_none' => $this->t('N/A')];
    $lines = explode(PHP_EOL, $config_options);
    foreach ($lines as $line) {
      $line = explode('|', $line);
      if ($line && isset($line[0]) && isset($line[1])) {
        $options[$line[0]] = $line[1];
      }
    }

    return $options;
  }

  /**
   * Helper function to get the index of the class at options list.
   *
   * @param string $name
   *   A config style name like background_color.
   * @param string $class
   *   The class name.
   *
   * @return int
   *   The index.
   */
  public function getStyleOptionIndexByClass(string $name, string $class) {
    $index = 0;
    $options = $this->getStyleOptions($name);
    $count = 0;
    foreach ($options as $key => $value) {
      if ($key == $class) {
        $index = $count;
        break;
      }
      $count++;
    }

    return $index;
  }

  /**
   * Helper function to get the options of given style name.
   *
   * @param string $name
   *   A config style name like background_color.
   *
   * @return array
   *   Array of key => value of style name options.
   */
  public function getStyleOptionsCount(string $name) {
    // -1 to drop the _none option from the count.
    $count = count($this->getStyleOptions($name)) - 1;
    return $count;
  }

  /**
   * Helper function to get the class from the options list.
   *
   * @param string $name
   *   A config style name like background_color.
   * @param int $index
   *   The index of the class at the option list.
   *
   * @return string
   *   The class name or null.
   */
  public function getStyleOptionClassByIndex(string $name, int $index) {
    $class = '';
    $options = $this->getStyleOptions($name);
    $count = 0;
    foreach ($options as $key => $value) {
      if ($count == $index) {
        $class = $key;
        break;
      }
      $count++;
    }

    return $class;
  }

}
