/**
 * @file
 * Environment initialization and core utilities for UtiliKit.
 *
 * This file provides the foundational environment setup for the UtiliKit
 * module, including configuration loading from drupalSettings, breakpoint
 * detection systems, logging utilities, and state management initialization.
 * It serves as the core foundation that other UtiliKit modules depend on
 * for consistent configuration access and environmental detection.
 *
 * Core Responsibilities:
 * - Global configuration setup from drupalSettings integration
 * - Responsive breakpoint detection and management system
 * - Development mode logging with configurable verbosity levels
 * - Environment state initialization and tracking
 * - Body class management for mode indication and feature flags
 * - Cross-browser compatibility and feature detection
 *
 * Configuration Management:
 * - Seamless integration with Drupal's drupalSettings system
 * - Fallback defaults for missing or incomplete configuration
 * - Runtime configuration validation and normalization
 * - Development vs production environment detection
 * - Feature flag management for experimental functionality
 *
 * Breakpoint System:
 * - Mobile-first responsive breakpoint detection
 * - Configurable breakpoint activation for performance optimization
 * - Real-time viewport width monitoring and breakpoint calculation
 * - Integration with CSS media query specifications
 * - Support for custom breakpoint configurations
 */
(function(Drupal, once, drupalSettings) {
  'use strict';

  // Ensure UtiliKit namespace exists for environment setup
  Drupal.utilikit = Drupal.utilikit || {};

  // ======================================================================= */
  // CONFIGURATION SETUP AND VALIDATION
  // ======================================================================= */

  /**
   * Core UtiliKit settings loaded from Drupal configuration.
   *
   * Provides centralized access to all UtiliKit configuration with fallback
   * defaults to ensure consistent behavior even when configuration is
   * incomplete or unavailable.
   */
  Drupal.utilikit.settings = drupalSettings.utilikit || {};

  /**
   * Development mode flag for debugging and development features.
   *
   * Controls the availability of development tools, enhanced logging,
   * and debugging interfaces throughout the UtiliKit system.
   */
  Drupal.utilikit.isDevMode = !!Drupal.utilikit.settings.devMode;

  /**
   * Page error display flag for development feedback.
   *
   * When enabled, shows validation errors and processing issues directly
   * on pages for immediate developer feedback during development workflows.
   */
  Drupal.utilikit.showPageErrors = !!Drupal.utilikit.settings.showPageErrors;

  /**
   * CSS transitions enablement for enhanced user experience.
   *
   * Controls whether UtiliKit adds smooth CSS transitions to utility
   * class changes for polished visual effects during dynamic updates.
   */
  Drupal.utilikit.enableTransitions = !!Drupal.utilikit.settings.enableTransitions;

  /**
   * Debounce delay for resize events in milliseconds.
   *
   * Controls how long to wait after resize events before triggering
   * UtiliKit's responsive recalculation to balance responsiveness with performance.
   */
  Drupal.utilikit.debounceMs = Drupal.utilikit.settings.debounce ?? 50;

  /**
   * Logging verbosity level for development and debugging.
   *
   * Controls the amount of console output generated by UtiliKit for
   * debugging and development purposes. Options: 'off', 'warnings', 'detailed'.
   */
  Drupal.utilikit.logLevel = Drupal.utilikit.settings.logLevel ?? 'warnings';

  /**
   * Admin preview mode for visual debugging of utility classes.
   *
   * When enabled, provides visual feedback and debugging aids for
   * utility class application and responsive behavior during development.
   */
  Drupal.utilikit.adminPreview = !!Drupal.utilikit.settings.adminPreview;

  /**
   * Current rendering mode (inline or static) for processing workflow.
   *
   * Determines whether UtiliKit generates CSS dynamically on the client
   * (inline) or uses pre-generated CSS files (static) for optimal performance.
   */
  Drupal.utilikit.renderingMode = Drupal.utilikit.settings.renderingMode ?? 'inline';

  /**
   * Active responsive breakpoints for utility class generation.
   *
   * Array of breakpoint names that are enabled for responsive utility
   * class variants. Disabled breakpoints are excluded for performance optimization.
   */
  Drupal.utilikit.activeBreakpoints = (drupalSettings.utilikit && drupalSettings.utilikit.activeBreakpoints)
    ? drupalSettings.utilikit.activeBreakpoints
    : ['sm', 'md', 'lg', 'xl', 'xxl'];

  // ======================================================================= */
  // RESPONSIVE BREAKPOINT SYSTEM
  // ======================================================================= */

  /**
   * Responsive breakpoint definitions in pixels (mobile-first approach).
   *
   * Defines the minimum viewport widths for each responsive breakpoint,
   * following a mobile-first approach that aligns with modern CSS
   * framework standards and responsive design best practices.
   *
   * Breakpoint Scale:
   * - sm (576px): Small devices (landscape phones)
   * - md (768px): Medium devices (tablets)
   * - lg (992px): Large devices (desktops)
   * - xl (1200px): Extra large devices (large desktops)
   * - xxl (1400px): Extra extra large devices (wide monitors)
   */
  Drupal.utilikit.breakpoints = {
    sm: 576,   // Small screens and up
    md: 768,   // Medium screens and up
    lg: 992,   // Large screens and up
    xl: 1200,  // Extra large screens and up
    xxl: 1400, // Extra extra large screens and up
  };

  /**
   * Determines the current active breakpoint based on viewport width.
   *
   * This function calculates which breakpoint is currently active by
   * comparing the current viewport width against the defined breakpoint
   * thresholds. It follows a mobile-first approach where smaller
   * breakpoints are the baseline and larger breakpoints progressively
   * enhance the experience.
   *
   * The function returns the largest breakpoint that the current viewport
   * width meets or exceeds, or an empty string for viewports smaller
   * than the smallest defined breakpoint (base/mobile).
   *
   * @returns {string}
   *   The current active breakpoint name ('sm', 'md', 'lg', 'xl', 'xxl')
   *   or empty string for base mobile viewport (< 576px).
   *
   * @example
   * // Get current breakpoint
   * const currentBp = Drupal.utilikit.getBreakpoint();
   * console.log('Current breakpoint:', currentBp); // 'md' for 800px width
   *
   * @example
   * // Use in responsive logic
   * if (Drupal.utilikit.getBreakpoint() === 'lg') {
   *   // Apply large screen specific behavior
   * }
   *
   * @example
   * // Check if at least medium size
   * const isAtLeastMedium = ['md', 'lg', 'xl', 'xxl'].includes(Drupal.utilikit.getBreakpoint());
   */
  Drupal.utilikit.getBreakpoint = function() {
    const width = window.innerWidth;
    if (width >= Drupal.utilikit.breakpoints.xxl) return 'xxl';
    if (width >= Drupal.utilikit.breakpoints.xl) return 'xl';
    if (width >= Drupal.utilikit.breakpoints.lg) return 'lg';
    if (width >= Drupal.utilikit.breakpoints.md) return 'md';
    if (width >= Drupal.utilikit.breakpoints.sm) return 'sm';
    return ''; // Base/mobile breakpoint (< 576px)
  };

  // ======================================================================= */
  // LOGGING AND DEBUGGING SYSTEM
  // ======================================================================= */

  /**
   * Centralized logging utility with configurable verbosity and development mode integration.
   *
   * This function provides consistent logging throughout the UtiliKit system
   * with configurable verbosity levels and development mode integration.
   * It ensures that logging only occurs in development environments and
   * respects user-configured log levels for optimal performance.
   *
   * The logging system supports structured data logging for enhanced
   * debugging capabilities and integrates seamlessly with browser
   * developer tools for comprehensive development workflow support.
   *
   * Log Levels:
   * - 'off': No logging output (production default)
   * - 'warnings': Warnings and errors only (development default)
   * - 'detailed': Comprehensive debug information (troubleshooting)
   *
   * @param {string} message
   *   The primary log message to display. Should be descriptive and
   *   provide clear context about the operation or event being logged.
   * @param {*} [data=null]
   *   Optional additional data to include with the log message.
   *   Can be any data type including objects, arrays, or primitive values.
   * @param {string} [type='log']
   *   The log message type, determining console output method and filtering.
   *   Supported values: 'log' for general information, 'warn' for warnings.
   *
   * @example
   * // Basic informational logging (detailed mode only)
   * Drupal.utilikit.utilikitLog('Processing element', null, 'log');
   *
   * @example
   * // Warning with additional context data
   * Drupal.utilikit.utilikitLog('Invalid class format', { className: 'uk-invalid' }, 'warn');
   *
   * @example
   * // Detailed logging with structured data
   * Drupal.utilikit.utilikitLog('Breakpoint changed', {
   *   from: 'md',
   *   to: 'lg',
   *   width: 1024,
   *   elementsProcessed: 42
   * }, 'log');
   */
  Drupal.utilikit.utilikitLog = function(message, data = null, type = 'log') {
    const isDevMode = Drupal.utilikit.isDevMode;
    const logLevel = Drupal.utilikit.logLevel ?? 'off';

    // Skip all logging if not in development mode or logging is disabled
    if (!isDevMode || logLevel === 'off') {
      return;
    }

    // Determine if this log message should be output based on level and type
    const allowLog = (
      (type === 'log' && logLevel === 'detailed') ||
      (type === 'warn' && (logLevel === 'warnings' || logLevel === 'detailed'))
    );

    if (!allowLog) {
      return;
    }

    // Consistent prefix for all UtiliKit log messages
    const prefix = '🛠️ UtiliKit:';

    // Output to appropriate console method based on message type
    if (type === 'log') {
      console.log(prefix, message, data ?? '');
    } else if (type === 'warn') {
      console.warn(prefix, message, data ?? '');
    }
  };

  // ======================================================================= */
  // ENVIRONMENT INITIALIZATION AND STATE MANAGEMENT
  // ======================================================================= */

  /**
   * Initializes the UtiliKit environment and establishes core system state.
   *
   * This function performs comprehensive environment setup including state
   * initialization, configuration validation, development mode setup, and
   * body class management for mode indication. It ensures that all UtiliKit
   * systems have access to consistent configuration and state management.
   *
   * Initialization Process:
   * 1. Create or validate core state management object
   * 2. Initialize tracking variables for responsive behavior
   * 3. Configure development mode features and logging
   * 4. Apply body classes for visual indication and feature flags
   * 5. Set up rendering mode specific configurations
   *
   * The function is designed to be safe to call multiple times, with
   * appropriate checks to prevent duplicate initialization while allowing
   * for state updates and reconfiguration as needed.
   *
   * @param {Document|Element} context
   *   The DOM context for environment initialization. Typically the
   *   document object for full page setup, or a specific container
   *   for scoped initialization after AJAX operations.
   *
   * @example
   * // Initialize environment for entire document
   * Drupal.utilikit.initEnvironment(document);
   *
   * @example
   * // Initialize for specific container after AJAX
   * const container = document.querySelector('.ajax-content');
   * Drupal.utilikit.initEnvironment(container);
   *
   * @example
   * // Reinitialize after configuration changes
   * // Update drupalSettings.utilikit configuration, then:
   * Drupal.utilikit.initEnvironment(document);
   */
  Drupal.utilikit.initEnvironment = function(context) {
    // Initialize or validate core state management object
    if (!Drupal.utilikit.state || !Drupal.utilikit.state.initialized) {
      Drupal.utilikit.state = {
        resizeTimeout: null,        // Debounce timer for resize events
        resizeFrame: null,          // Animation frame for smooth updates
        lastWidth: window.innerWidth, // Track viewport changes
        lastBreakpoint: Drupal.utilikit.getBreakpoint(), // Track breakpoint changes
        resizeListenerAttached: false, // Prevent duplicate listeners
        initialized: true           // Initialization completion flag
      };
    }

    // Development mode initialization and configuration logging
    if (Drupal.utilikit.isDevMode && !Drupal.utilikit.logged) {
      Drupal.utilikit.utilikitLog?.('Settings loaded', {
        isDevMode: Drupal.utilikit.isDevMode,
        showPageErrors: Drupal.utilikit.showPageErrors,
        enableTransitions: Drupal.utilikit.enableTransitions,
        debounceMs: Drupal.utilikit.debounceMs,
        logLevel: Drupal.utilikit.logLevel,
        adminPreview: Drupal.utilikit.adminPreview,
        activeBreakpoints: Drupal.utilikit.activeBreakpoints,
        breakpoints: Drupal.utilikit.breakpoints
      });
      Drupal.utilikit.logged = true; // Prevent duplicate logging
    }

    // Apply admin preview mode body class for visual debugging
    if (Drupal.utilikit.adminPreview && !document.body.classList.contains('uk-admin')) {
      document.body.classList.add('uk-admin');
    }

    // Apply transitions enablement body class for CSS integration
    if (Drupal.utilikit.enableTransitions && !document.body.classList.contains('uk-trans')) {
      document.body.classList.add('uk-trans');
    }

    // Apply rendering mode specific body classes for CSS targeting
    if (Drupal.utilikit.renderingMode === 'static') {
      document.body.classList.remove('utilikit-inline-mode');
      document.body.classList.add('utilikit-static-mode');
      Drupal.utilikit.utilikitLog('Body class added: utilikit-static-mode', null, 'log');
    } else {
      document.body.classList.remove('utilikit-static-mode');
      document.body.classList.add('utilikit-inline-mode');
      Drupal.utilikit.utilikitLog('Body class added: utilikit-inline-mode', null, 'log');
    }
  };

})(Drupal, once, drupalSettings);

/**
 * Integration Notes for UtiliKit Environment System:
 *
 * Configuration Architecture:
 * - Seamless integration with Drupal's drupalSettings system
 * - Comprehensive fallback defaults for missing configuration values
 * - Runtime validation and normalization of configuration data
 * - Support for dynamic configuration updates during runtime
 * - Feature flag management for experimental and beta functionality
 *
 * Responsive Breakpoint Integration:
 * - Mobile-first breakpoint system aligned with CSS framework standards
 * - Real-time breakpoint detection using viewport width monitoring
 * - Configurable breakpoint activation for performance optimization
 * - Integration with CSS media query specifications for consistency
 * - Support for custom breakpoint configurations and extensions
 *
 * Development and Production Environments:
 * - Comprehensive development mode with enhanced debugging capabilities
 * - Production-safe defaults with minimal overhead and no debug output
 * - Configurable logging levels for different development scenarios
 * - Visual debugging aids through body class management
 * - Development tool integration with browser developer consoles
 *
 * State Management and Lifecycle:
 * - Centralized state object for tracking runtime environment status
 * - Initialization safety with duplicate prevention and validation
 * - Integration with resize handling and responsive behavior systems
 * - Support for state reset and reinitialization during development
 * - Cross-component state sharing for consistent behavior
 *
 * Performance Optimization Features:
 * - Lazy initialization with minimal startup overhead
 * - Efficient breakpoint calculation with caching considerations
 * - Conditional feature activation based on configuration flags
 * - Development feature exclusion in production builds
 * - Optimized logging with level-based filtering for performance
 *
 * Browser Compatibility and Standards:
 * - Cross-browser viewport width detection using standard APIs
 * - Progressive enhancement with graceful degradation support
 * - Standards-compliant breakpoint definitions and calculations
 * - Modern JavaScript features with fallback compatibility
 * - Integration with browser developer tools for debugging
 *
 * Error Handling and Robustness:
 * - Graceful handling of missing or corrupted configuration
 * - Fallback behavior for incomplete drupalSettings integration
 * - Non-blocking initialization that preserves other functionality
 * - Comprehensive validation of environment setup and dependencies
 * - Development mode error reporting with actionable feedback
 */
