<?php

declare(strict_types=1);

namespace Drupal\Tests\utilikit\Unit\Service;

use Drupal\Tests\UnitTestCase;
use Drupal\utilikit\Service\UtilikitCssGenerator;
use Drupal\utilikit\Service\UtilikitCssParser;
use Drupal\utilikit\Service\UtilikitStateManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the UtilikitCssGenerator service.
 *
 * @group utilikit
 * @coversDefaultClass \Drupal\utilikit\Service\UtilikitCssGenerator
 */
class UtilikitCssGeneratorTest extends UnitTestCase {

  /**
   * The CSS generator service under test.
   *
   * @var \Drupal\utilikit\Service\UtilikitCssGenerator
   */
  protected UtilikitCssGenerator $cssGenerator;

  /**
   * Mock CSS parser.
   *
   * @var \Drupal\utilikit\Service\UtilikitCssParser&\PHPUnit\Framework\MockObject\MockObject
   */
  protected UtilikitCssParser&MockObject $cssParser;

  /**
   * Mock state manager.
   *
   * @var \Drupal\utilikit\Service\UtilikitStateManager&\PHPUnit\Framework\MockObject\MockObject
   */
  protected UtilikitStateManager&MockObject $stateManager;

  /**
   * Mock config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected ConfigFactoryInterface&MockObject $configFactory;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected LoggerInterface&MockObject $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->cssParser = $this->createMock(UtilikitCssParser::class);
    $this->stateManager = $this->createMock(UtilikitStateManager::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    // Setup default config.
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['css_optimization', TRUE],
        ['css_minify', TRUE],
        ['css_comments', FALSE],
        ['media_queries', []],
        ['css_prefix', ''],
      ]);

    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('utilikit.settings')
      ->willReturn($config);

    $this->cssGenerator = new UtilikitCssGenerator(
      $this->cssParser,
      $this->stateManager,
      $this->configFactory,
      $this->logger
    );
  }

  /**
   * Tests generateCss with simple classes.
   *
   * @covers ::generateCss
   */
  public function testGenerateCssSimple(): void {
    $classes = [
      '.uk-pd--20' => 'padding: 20px;',
      '.uk-mg--10' => 'margin: 10px;',
    ];

    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willReturn($classes);

    $css = $this->cssGenerator->generateCss();

    $this->assertStringContainsString('.uk-pd--20', $css);
    $this->assertStringContainsString('padding: 20px;', $css);
    $this->assertStringContainsString('.uk-mg--10', $css);
    $this->assertStringContainsString('margin: 10px;', $css);
  }

  /**
   * Tests generateCss with minification enabled.
   *
   * @covers ::generateCss
   * @covers ::minifyCss
   */
  public function testGenerateCssMinified(): void {
    $classes = [
      '.uk-pd--20' => 'padding: 20px;',
      '.uk-mg--10' => 'margin: 10px;',
    ];

    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willReturn($classes);

    $css = $this->cssGenerator->generateCss();

    // Minified CSS should not contain extra spaces or newlines.
    $this->assertStringNotContainsString('  ', $css);
    $this->assertStringContainsString('.uk-pd--20{padding:20px}', $css);
    $this->assertStringContainsString('.uk-mg--10{margin:10px}', $css);
  }

  /**
   * Tests generateCss with comments enabled.
   *
   * @covers ::generateCss
   */
  public function testGenerateCssWithComments(): void {
    // Setup config with comments enabled.
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['css_optimization', TRUE],
        ['css_minify', FALSE],
        ['css_comments', TRUE],
        ['media_queries', []],
        ['css_prefix', ''],
      ]);

    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('utilikit.settings')
      ->willReturn($config);

    $classes = [
      '.uk-pd--20' => 'padding: 20px;',
    ];

    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willReturn($classes);

    $css = $this->cssGenerator->generateCss();

    $this->assertStringContainsString('/* Generated by UtiliKit', $css);
    $this->assertStringContainsString('/* Class: uk-pd--20 */', $css);
  }

  /**
   * Tests generateCss with media queries.
   *
   * @covers ::generateCss
   * @covers ::applyMediaQueries
   */
  public function testGenerateCssWithMediaQueries(): void {
    // Setup config with media queries.
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['css_optimization', TRUE],
        ['css_minify', FALSE],
        ['css_comments', FALSE],
        ['media_queries', [
          'sm' => '(min-width: 640px)',
          'md' => '(min-width: 768px)',
          'lg' => '(min-width: 1024px)',
        ]],
        ['css_prefix', ''],
      ]);

    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('utilikit.settings')
      ->willReturn($config);

    $classes = [
      '.uk-pd--20' => 'padding: 20px;',
      '.uk-pd-sm--20' => 'padding: 20px;',
      '.uk-pd-md--20' => 'padding: 20px;',
      '.uk-pd-lg--20' => 'padding: 20px;',
    ];

    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willReturn($classes);

    $css = $this->cssGenerator->generateCss();

    $this->assertStringContainsString('.uk-pd--20', $css);
    $this->assertStringContainsString('@media (min-width: 640px)', $css);
    $this->assertStringContainsString('@media (min-width: 768px)', $css);
    $this->assertStringContainsString('@media (min-width: 1024px)', $css);
  }

  /**
   * Tests generateCss with CSS prefix.
   *
   * @covers ::generateCss
   */
  public function testGenerateCssWithPrefix(): void {
    // Setup config with CSS prefix.
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['css_optimization', TRUE],
        ['css_minify', FALSE],
        ['css_comments', FALSE],
        ['media_queries', []],
        ['css_prefix', '.utilikit-wrapper'],
      ]);

    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('utilikit.settings')
      ->willReturn($config);

    $classes = [
      '.uk-pd--20' => 'padding: 20px;',
      '.uk-mg--10' => 'margin: 10px;',
    ];

    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willReturn($classes);

    $css = $this->cssGenerator->generateCss();

    $this->assertStringContainsString('.utilikit-wrapper .uk-pd--20', $css);
    $this->assertStringContainsString('.utilikit-wrapper .uk-mg--10', $css);
  }

  /**
   * Tests generateForClasses method.
   *
   * @covers ::generateForClasses
   */
  public function testGenerateForClasses(): void {
    $classes = ['uk-pd--20', 'uk-mg--10', 'uk-text--center'];

    $this->cssParser->expects($this->exactly(3))
      ->method('parseClass')
      ->willReturnOnConsecutiveCalls(
        'padding: 20px;',
        'margin: 10px;',
        'text-align: center;'
      );

    $css = $this->cssGenerator->generateForClasses($classes);

    $this->assertStringContainsString('.uk-pd--20', $css);
    $this->assertStringContainsString('padding: 20px;', $css);
    $this->assertStringContainsString('.uk-mg--10', $css);
    $this->assertStringContainsString('margin: 10px;', $css);
    $this->assertStringContainsString('.uk-text--center', $css);
    $this->assertStringContainsString('text-align: center;', $css);
  }

  /**
   * Tests generateForClasses with invalid classes.
   *
   * @covers ::generateForClasses
   */
  public function testGenerateForClassesWithInvalid(): void {
    $classes = ['uk-pd--20', 'invalid-class'];

    $this->cssParser->expects($this->exactly(2))
      ->method('parseClass')
      ->willReturnOnConsecutiveCalls(
        'padding: 20px;',
        NULL
      );

    $this->logger->expects($this->once())
      ->method('warning')
      ->with($this->stringContains('Failed to parse class'));

    $css = $this->cssGenerator->generateForClasses($classes);

    $this->assertStringContainsString('.uk-pd--20', $css);
    $this->assertStringNotContainsString('invalid-class', $css);
  }

  /**
   * Tests optimizeCss method.
   *
   * @covers ::optimizeCss
   */
  public function testOptimizeCss(): void {
    $input = "
      .uk-pd--20 {
        padding: 20px;
      }

      .uk-mg--10 {
        margin: 10px;
      }

      /* Comment */
      .uk-text--center {
        text-align: center;
      }
    ";

    $optimized = $this->cssGenerator->optimizeCss($input);

    // Should remove extra whitespace and comments.
    $this->assertStringNotContainsString('/* Comment */', $optimized);
    $this->assertStringNotContainsString("\n\n", $optimized);
    $this->assertStringContainsString('.uk-pd--20', $optimized);
    $this->assertStringContainsString('.uk-mg--10', $optimized);
    $this->assertStringContainsString('.uk-text--center', $optimized);
  }

  /**
   * Tests groupRelatedClasses method.
   *
   * @covers ::groupRelatedClasses
   */
  public function testGroupRelatedClasses(): void {
    $classes = [
      '.uk-pd--20' => 'padding: 20px;',
      '.uk-pd--10' => 'padding: 10px;',
      '.uk-mg--20' => 'margin: 20px;',
      '.uk-mg--10' => 'margin: 10px;',
      '.uk-text--center' => 'text-align: center;',
      '.uk-text--left' => 'text-align: left;',
    ];

    $grouped = $this->cssGenerator->groupRelatedClasses($classes);

    $this->assertArrayHasKey('pd', $grouped);
    $this->assertArrayHasKey('mg', $grouped);
    $this->assertArrayHasKey('text', $grouped);
    $this->assertCount(2, $grouped['pd']);
    $this->assertCount(2, $grouped['mg']);
    $this->assertCount(2, $grouped['text']);
  }

  /**
   * Tests generateWithOptions method.
   *
   * @covers ::generateWithOptions
   */
  public function testGenerateWithOptions(): void {
    $classes = [
      '.uk-pd--20' => 'padding: 20px;',
      '.uk-mg--10' => 'margin: 10px;',
    ];

    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willReturn($classes);

    $options = [
      'minify' => FALSE,
      'comments' => TRUE,
      'optimize' => FALSE,
      'prefix' => '.custom-wrapper',
    ];

    $css = $this->cssGenerator->generateWithOptions($options);

    $this->assertStringContainsString('/* Generated by UtiliKit', $css);
    $this->assertStringContainsString('.custom-wrapper .uk-pd--20', $css);
    $this->assertStringContainsString('padding: 20px;', $css);
    $this->assertStringNotContainsString('{padding:20px}', $css); // Not minified.
  }

  /**
   * Tests empty class handling.
   *
   * @covers ::generateCss
   */
  public function testGenerateCssEmpty(): void {
    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willReturn([]);

    $css = $this->cssGenerator->generateCss();

    $this->assertIsString($css);
    $this->assertEmpty(trim($css));
  }

  /**
   * Tests error handling in generation.
   *
   * @covers ::generateCss
   */
  public function testGenerateCssErrorHandling(): void {
    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willThrowException(new \Exception('Database error'));

    $this->logger->expects($this->once())
      ->method('error')
      ->with($this->stringContains('Error generating CSS'));

    $css = $this->cssGenerator->generateCss();

    $this->assertIsString($css);
    $this->assertEmpty($css);
  }

  /**
   * Tests CSS validation.
   *
   * @covers ::validateCss
   */
  public function testValidateCss(): void {
    // Valid CSS.
    $validCss = '.uk-pd--20 { padding: 20px; } .uk-mg--10 { margin: 10px; }';
    $this->assertTrue($this->cssGenerator->validateCss($validCss));

    // Invalid CSS.
    $invalidCss = '.uk-pd--20 { padding: 20px';
    $this->assertFalse($this->cssGenerator->validateCss($invalidCss));

    // Empty CSS.
    $this->assertTrue($this->cssGenerator->validateCss(''));
  }

  /**
   * Tests CSS statistics generation.
   *
   * @covers ::getStatistics
   */
  public function testGetStatistics(): void {
    $classes = [
      '.uk-pd--20' => 'padding: 20px;',
      '.uk-pd--10' => 'padding: 10px;',
      '.uk-mg--20' => 'margin: 20px;',
      '.uk-text--center' => 'text-align: center;',
    ];

    $this->stateManager->expects($this->once())
      ->method('getStoredClasses')
      ->willReturn($classes);

    $stats = $this->cssGenerator->getStatistics();

    $this->assertArrayHasKey('total_classes', $stats);
    $this->assertArrayHasKey('css_size', $stats);
    $this->assertArrayHasKey('minified_size', $stats);
    $this->assertArrayHasKey('compression_ratio', $stats);
    $this->assertArrayHasKey('property_breakdown', $stats);
    $this->assertEquals(4, $stats['total_classes']);
  }

}
