<?php

declare(strict_types=1);

namespace Drupal\ux_enhanced_autocomplete\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for UX Enhanced Autocomplete module.
 */
class ConfigurationForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ux_enhanced_autocomplete.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ux_enhanced_autocomplete_configuration_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('ux_enhanced_autocomplete.settings');

    $form['general'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('General settings'),
    ];

    $form['general']['match_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum number of matches'),
      '#description' => $this->t('Maximum number of suggestions to display in the autocomplete dropdown.'),
      '#default_value' => $config->get('match_limit') ?? 20,
      '#min' => 1,
      '#max' => 100,
      '#required' => TRUE,
    ];

    $form['general']['min_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum search length'),
      '#description' => $this->t('Minimum number of characters required before autocomplete suggestions appear.'),
      '#default_value' => $config->get('min_length') ?? 3,
      '#min' => 1,
      '#max' => 10,
      '#required' => TRUE,
    ];

    // Entity type/bundle options.
    $form['entity_type'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Entity type/bundle display'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $form['entity_type']['show_entity_type'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show entity type/bundle'),
      '#description' => $this->t('Display the entity type or bundle name in autocomplete results.'),
      '#default_value' => $config->get('show_entity_type') ?? TRUE,
    ];

    $form['entity_type']['bold_entity_type'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Bold entity type/bundle'),
      '#description' => $this->t('Make the entity type or bundle name bold.'),
      '#default_value' => $config->get('bold_entity_type') ?? FALSE,
      '#states' => [
        'visible' => [
          ':input[name="show_entity_type"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Entity ID options.
    $form['entity_id'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Entity ID display'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $form['entity_id']['show_entity_id'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show entity ID'),
      '#description' => $this->t('Display the entity ID in autocomplete results.'),
      '#default_value' => $config->get('show_entity_id') ?? TRUE,
    ];

    $form['entity_id']['bold_entity_id'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Bold entity ID'),
      '#description' => $this->t('Make the entity ID bold.'),
      '#default_value' => $config->get('bold_entity_id') ?? FALSE,
      '#states' => [
        'visible' => [
          ':input[name="show_entity_id"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Date options.
    $form['date'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Date display'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $form['date']['show_date'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show date'),
      '#description' => $this->t('Display creation/modification date in autocomplete results.'),
      '#default_value' => $config->get('show_date') ?? TRUE,
    ];

    $form['date']['bold_date'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Bold date'),
      '#description' => $this->t('Make the date bold.'),
      '#default_value' => $config->get('bold_date') ?? FALSE,
      '#states' => [
        'visible' => [
          ':input[name="show_date"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['date']['date_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Date format'),
      '#description' => $this->t('Format to use when displaying dates.'),
      '#options' => [
        'short' => $this->t('Short (e.g., 12/31/2023)'),
        'medium' => $this->t('Medium (e.g., Dec 31, 2023)'),
        'long' => $this->t('Long (e.g., December 31, 2023)'),
      ],
      '#default_value' => $config->get('date_format') ?? 'medium',
      '#states' => [
        'visible' => [
          ':input[name="show_date"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Author options.
    $form['author'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Author display'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $form['author']['show_author'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show author'),
      '#description' => $this->t('Display the author name in autocomplete results.'),
      '#default_value' => $config->get('show_author') ?? TRUE,
    ];

    $form['author']['bold_author'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Bold author'),
      '#description' => $this->t('Make the author name bold.'),
      '#default_value' => $config->get('bold_author') ?? FALSE,
      '#states' => [
        'visible' => [
          ':input[name="show_author"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Formatting options.
    $form['formatting'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Formatting options'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $form['formatting']['separator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Information separator'),
      '#description' => $this->t('Character(s) used to separate information elements in autocomplete results.'),
      '#default_value' => $config->get('separator') ?? ' | ',
      '#size' => 10,
      '#maxlength' => 10,
    ];

    $form['formatting']['enable_color'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable accent color'),
      '#description' => $this->t('Apply an accent color to highlight certain elements in autocomplete results.'),
      '#default_value' => $config->get('enable_color') ?? FALSE,
    ];

    $form['formatting']['accent_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Accent color'),
      '#description' => $this->t('Color used to highlight entity type and ID in autocomplete results.'),
      '#default_value' => $config->get('accent_color') ?? '#0073aa',
      '#states' => [
        'visible' => [
          ':input[name="enable_color"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['formatting']['truncate_title'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Truncate long titles'),
      '#description' => $this->t('Truncate entity titles that are longer than the specified character limit.'),
      '#default_value' => $config->get('truncate_title') ?? TRUE,
    ];

    $form['formatting']['title_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum title length'),
      '#description' => $this->t('Maximum number of characters to display for entity titles before truncating.'),
      '#default_value' => $config->get('title_max_length') ?? 60,
      '#min' => 20,
      '#max' => 200,
      '#step' => 1,
      '#field_suffix' => $this->t('characters'),
      '#states' => [
        'visible' => [
          ':input[name="truncate_title"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('ux_enhanced_autocomplete.settings')
      ->set('match_limit', $form_state->getValue('match_limit'))
      ->set('min_length', $form_state->getValue('min_length'))
      ->set('separator', $form_state->getValue('separator'))
      ->set('show_entity_type', $form_state->getValue('show_entity_type'))
      ->set('bold_entity_type', $form_state->getValue('bold_entity_type'))
      ->set('show_entity_id', $form_state->getValue('show_entity_id'))
      ->set('bold_entity_id', $form_state->getValue('bold_entity_id'))
      ->set('show_date', $form_state->getValue('show_date'))
      ->set('bold_date', $form_state->getValue('bold_date'))
      ->set('show_author', $form_state->getValue('show_author'))
      ->set('bold_author', $form_state->getValue('bold_author'))
      ->set('date_format', $form_state->getValue('date_format'))
      ->set('enable_color', $form_state->getValue('enable_color'))
      ->set('accent_color', $form_state->getValue('accent_color'))
      ->set('truncate_title', $form_state->getValue('truncate_title'))
      ->set('title_max_length', $form_state->getValue('title_max_length'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
