<?php

declare(strict_types=1);

namespace Drupal\ux_enhanced_autocomplete\EntityAutoComplete;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\Tags;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityAutocompleteMatcherInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManager;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Enhanced entity autocomplete matcher with improved display.
 *
 * Provides a better user experience for entity reference autocomplete fields
 * by displaying additional contextual information such as entity type, ID,
 * creation/modification date, and author information. All display options
 * are configurable through the module's configuration form.
 */
final class UxEnhancedEntityAutocompleteMatcher implements EntityAutocompleteMatcherInterface {

  use StringTranslationTrait;

  /**
   * The default minimum length of the input string to trigger autocomplete.
   */
  public const int DEFAULT_AUTOCOMPLETE_MIN_LENGTH = 1;

  /**
   * The default maximum number of matches to return.
   */
  public const int DEFAULT_MATCH_LIMIT = 20;

  /**
   * The default separator for information parts.
   */
  public const string DEFAULT_SEPARATOR = ' | ';

  /**
   * Supported entity types for enhanced autocomplete.
   */
  public const array SUPPORTED_ENTITY_TYPES = ['node', 'taxonomy_term'];

  /**
   * Constructs a new UxEnhancedEntityAutocompleteMatcher.
   *
   * @param \Drupal\Core\Entity\EntityAutocompleteMatcherInterface $original
   *   The original autocomplete matcher service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManager $selectionManager
   *   The entity reference selection plugin manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundleInfo
   *   The entity type bundle info service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger channel factory service.
   */
  public function __construct(
    protected EntityAutocompleteMatcherInterface $original,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected DateFormatterInterface $dateFormatter,
    protected SelectionPluginManager $selectionManager,
    protected ModuleHandlerInterface $moduleHandler,
    protected EntityTypeBundleInfoInterface $bundleInfo,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelFactoryInterface $loggerFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getMatches($target_type, $selection_handler, $selection_settings, $string = ''): array {
    // Fallback to original matcher if conditions are not met.
    if (!$this->shouldUseEnhancedMatcher($target_type, $string)) {
      return $this->original->getMatches($target_type, $selection_handler, $selection_settings, $string);
    }

    $config = $this->configFactory->get('ux_enhanced_autocomplete.settings');
    $options = $this->buildSelectionOptions($target_type, $selection_handler, $selection_settings, $config);

    try {
      $handler = $this->selectionManager->getInstance($options);
      $match_operator = $options['match_operator'] ?? 'CONTAINS';
      $match_limit = $config->get('match_limit') ?? self::DEFAULT_MATCH_LIMIT;

      $entity_labels = $handler->getReferenceableEntities($string, $match_operator, $match_limit);

      return $this->processEntityMatches($entity_labels, $target_type, $config);
    }
    catch (\Exception $e) {
      // Log error and fallback to original matcher.
      $this->loggerFactory->get('ux_enhanced_autocomplete')->error('Error in enhanced autocomplete: @message', ['@message' => $e->getMessage()]);
      return $this->original->getMatches($target_type, $selection_handler, $selection_settings, $string);
    }
  }

  /**
   * Determines if the enhanced matcher should be used.
   *
   * @param string $target_type
   *   The target entity type.
   * @param string $string
   *   The input string.
   *
   * @return bool
   *   TRUE if enhanced matcher should be used, FALSE otherwise.
   */
  protected function shouldUseEnhancedMatcher(string $target_type, string $string): bool {
    if (!$this->moduleHandler->moduleExists('ux_enhanced_autocomplete')) {
      return FALSE;
    }

    if (!in_array($target_type, self::SUPPORTED_ENTITY_TYPES, TRUE)) {
      return FALSE;
    }

    $config = $this->configFactory->get('ux_enhanced_autocomplete.settings');
    $min_length = $config->get('min_length') ?? self::DEFAULT_AUTOCOMPLETE_MIN_LENGTH;

    return !empty($string) && strlen($string) >= $min_length;
  }

  /**
   * Builds selection options for the entity reference handler.
   *
   * @param string $target_type
   *   The target entity type.
   * @param string $selection_handler
   *   The selection handler name.
   * @param array $selection_settings
   *   The selection settings.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   *
   * @return array
   *   The selection options array.
   */
  protected function buildSelectionOptions(string $target_type, string $selection_handler, array $selection_settings, ImmutableConfig $config): array {
    $options = $selection_settings + [
      'target_type' => $target_type,
      'handler' => $selection_handler,
    ];

    // Set default sort order based on entity type.
    $sort_field = match ($target_type) {
      'node' => 'title',
      'taxonomy_term' => 'name',
      default => NULL,
    };

    if ($sort_field) {
      $options['sort'] = [
        'field' => $sort_field,
        'direction' => 'ASC',
      ];
    }

    $options['match_limit'] = $config->get('match_limit') ?? self::DEFAULT_MATCH_LIMIT;

    return $options;
  }

  /**
   * Processes entity matches into the expected autocomplete format.
   *
   * @param array $entity_labels
   *   The entity labels from the selection handler.
   * @param string $target_type
   *   The target entity type.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   *
   * @return array
   *   The processed matches array.
   */
  protected function processEntityMatches(array $entity_labels, string $target_type, ImmutableConfig $config): array {
    $matches = [];

    foreach ($entity_labels as $values) {
      foreach ($values as $entity_id => $label) {
        $entity = $this->loadEntity($target_type, $entity_id);

        if (!$entity) {
          continue;
        }

        $bundle_label = $this->getBundleLabel($entity, $target_type);
        $formatted_output = $this->buildFormattedOutput($entity, $bundle_label, $config);
        $selection_key = $this->buildSelectionKey($entity);

        $matches[] = [
          'value' => $selection_key,
          'label' => $formatted_output,
        ];
      }
    }

    return $matches;
  }

  /**
   * Loads an entity safely.
   *
   * @param string $target_type
   *   The entity type.
   * @param mixed $entity_id
   *   The entity ID.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The loaded entity or NULL if not found.
   */
  protected function loadEntity(string $target_type, mixed $entity_id): ?EntityInterface {
    try {
      return $this->entityTypeManager->getStorage($target_type)->load($entity_id);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ux_enhanced_autocomplete')->warning('Failed to load entity @type:@id: @message', [
        '@type' => $target_type,
        '@id' => $entity_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets the human-readable bundle label for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $target_type
   *   The target entity type.
   *
   * @return string
   *   The bundle label.
   */
  protected function getBundleLabel(EntityInterface $entity, string $target_type): string {
    $bundles = $this->bundleInfo->getBundleInfo($target_type);
    $bundle_label = $bundles[$entity->bundle()]['label'] ?? $entity->bundle();

    return $bundle_label instanceof TranslatableMarkup
      ? $bundle_label->render()
      : (string) $bundle_label;
  }

  /**
   * Builds the selection key for the autocomplete result.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string
   *   The encoded selection key.
   */
  protected function buildSelectionKey(EntityInterface $entity): string {
    $display_text = sprintf('%s (%d)', $entity->label(), $entity->id());
    $key = preg_replace('/\s\s+/', ' ', str_replace("\n", '', trim(Html::decodeEntities(strip_tags($display_text)))));

    return Tags::encode($key);
  }

  /**
   * Builds the formatted HTML output for autocomplete suggestion.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to format.
   * @param string $bundle_label
   *   The human-readable bundle label.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   *
   * @return string
   *   Formatted HTML string.
   */
  protected function buildFormattedOutput(EntityInterface $entity, string $bundle_label, ImmutableConfig $config): string {
    $info_parts = [];
    $separator = $config->get('separator') ?? self::DEFAULT_SEPARATOR;
    $enable_color = (bool) ($config->get('enable_color') ?? FALSE);
    $accent_color = $config->get('accent_color') ?? '#045e7c';

    // Add entity type/bundle information.
    if ($config->get('show_entity_type') ?? TRUE) {
      $info_parts[] = $this->formatInfoPart(
        $bundle_label,
        (bool) ($config->get('bold_entity_type') ?? FALSE),
        $enable_color ? $accent_color : NULL
      );
    }

    // Add entity ID information.
    if ($config->get('show_entity_id') ?? TRUE) {
      $info_parts[] = $this->formatInfoPart(
        '#' . $entity->id(),
        (bool) ($config->get('bold_entity_id') ?? FALSE),
        $enable_color ? $accent_color : NULL
      );
    }

    // Add date information.
    if ($config->get('show_date') ?? TRUE) {
      $date_text = $this->getEntityDateText($entity, $config);
      if ($date_text) {
        $info_parts[] = $this->formatInfoPart(
          $date_text,
          (bool) ($config->get('bold_date') ?? FALSE)
        );
      }
    }

    // Add author information.
    if ($config->get('show_author') ?? TRUE) {
      $author_text = $this->getEntityAuthorText($entity);
      if ($author_text) {
        $info_parts[] = $this->formatInfoPart(
          $author_text,
          (bool) ($config->get('bold_author') ?? FALSE)
        );
      }
    }

    $title = $this->formatTitle($entity->label(), $config);
    $info = !empty($info_parts) ? implode($separator, $info_parts) : '';

    if ($info) {
      return '<div class="ux-enhanced-autocomplete-item">' . $title . '<br><small>' . $info . '</small></div>';
    }

    return '<div class="ux-enhanced-autocomplete-item">' . $title . '</div>';
  }

  /**
   * Formats an information part with optional bold styling and color.
   *
   * @param string $text
   *   The text to format.
   * @param bool $bold
   *   Whether to make the text bold.
   * @param string|null $color
   *   Optional color to apply as inline style.
   *
   * @return string
   *   The formatted text.
   */
  protected function formatInfoPart(string $text, bool $bold, ?string $color = NULL): string {
    $escaped_text = Html::escape($text);

    // Apply bold formatting if requested.
    if ($bold) {
      $escaped_text = '<strong>' . $escaped_text . '</strong>';
    }

    // Apply color styling if provided.
    if ($color) {
      $style = 'color: ' . Html::escape($color) . ';';
      $escaped_text = '<span style="' . $style . '">' . $escaped_text . '</span>';
    }

    return $escaped_text;
  }

  /**
   * Gets the formatted date text for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   *
   * @return string|null
   *   The formatted date text or NULL if not available.
   */
  protected function getEntityDateText(EntityInterface $entity, ImmutableConfig $config): ?string {
    $timestamp = match ($entity->getEntityTypeId()) {
      'taxonomy_term' => method_exists($entity, 'getChangedTime') ? $entity->getChangedTime() : NULL,
      'node' => method_exists($entity, 'getCreatedTime') ? $entity->getCreatedTime() : NULL,
      default => NULL,
    };

    if (!$timestamp) {
      return NULL;
    }

    $date_format = $config->get('date_format') ?? 'medium';
    return $this->dateFormatter->format($timestamp, $date_format);
  }

  /**
   * Gets the author text for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string|null
   *   The author text or NULL if not available.
   */
  protected function getEntityAuthorText(EntityInterface $entity): ?string {
    if (!method_exists($entity, 'getOwner')) {
      return NULL;
    }

    $owner = $entity->getOwner();
    return $owner ? $owner->label() : NULL;
  }

  /**
   * Formats the entity title with optional truncation.
   *
   * @param string $title
   *   The entity title.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   *
   * @return string
   *   The formatted and escaped title.
   */
  protected function formatTitle(string $title, ImmutableConfig $config): string {
    $truncate_title = (bool) ($config->get('truncate_title') ?? TRUE);
    $max_length = (int) ($config->get('title_max_length') ?? 60);

    // Truncate the title if enabled and it's longer than the max length.
    if ($truncate_title && mb_strlen($title) > $max_length) {
      $title = Unicode::truncate($title, $max_length, TRUE, TRUE);
    }

    return Html::escape($title);
  }

}
