<?php

namespace Drupal\vais_promos\Plugin\SearchResults;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\vertex_ai_search\Plugin\VertexSearchResultsPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides simple autocomplete based on node title and/or description.
 *
 * @VertexSearchResultsPlugin(
 *   id = "vertex_search_results_promotions",
 *   title = @Translation("Adding Promotions to Search Results"),
 *   weight = 0
 * )
 */
class PromotionSearchResults extends VertexSearchResultsPluginBase {

  /**
   * Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $eTypeManager;

  /**
   * Database Connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Path Alias Manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * Constructor.
   *
   * @param array $configuration
   *   Configuration array containing information about search page.
   * @param string $plugin_id
   *   Identifier of custom plugin.
   * @param array $plugin_definition
   *   Provides definition of search plugin.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The Entity Type Manager service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection service.
   * @param \Drupal\path_alias\AliasManagerInterface $aliasManager
   *   The path alias manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
    Connection $database,
    AliasManagerInterface $aliasManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->eTypeManager = $entityTypeManager;
    $this->database = $database;
    $this->aliasManager = $aliasManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('path_alias.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function retrieveCuratedResults(array $search_page_config, array $search_parameters) {

    $promotions = $this->retrievePromotions($search_page_config['id'], $search_parameters['keys']);

    return $promotions;

  }

  /**
   * {@inheritdoc}
   */
  public function manipulatePageResults(array $search_page_config, array $search_parameters, array $search_results, array $curated_content) {

    return $this->addPromotions($search_parameters['keys'], $search_results, $curated_content);

  }

  /**
   * {@inheritdoc}
   */
  public function modifyPageResults(string $keyword, array $searchResults, string $search_page_id) {

    $promotions = $this->retrievePromotions($search_page_id, $keyword);

    return $this->addPromotions($keyword, $searchResults, $promotions);

  }

  /**
   * Helper function to merge promotions and results.
   *
   * @param string $keyword
   *   Keywords used in the search.
   * @param array $search_results
   *   Search results in an array.
   * @param array $promotions
   *   Promoted content to add to search results.
   *
   * @return array
   *   Array blending promotions with search results.
   */
  private function addPromotions(string $keyword, array $search_results, array $promotions) {

    foreach ($promotions as $promotion) {
      // Create the promotion result.
      $promo = [
        '#theme' => 'vais_promos_promotion',
        '#title' => $promotion['title'],
        '#description' => $promotion['description'],
        '#promo_type' => $promotion['type'],
        '#href' => $promotion['reference'],
        '#term' => $keyword,
        '#plugin_id' => $promotion['plugin_id'],
        '#attached' => [
          'library' => [
            'vais_promos/vaisPromos',
          ],
        ],
      ];

      // Inserts promotion after any messages that may exist.
      foreach ($search_results as $key => $search_result) {

        if (!empty($search_result['#result'])) {
          array_splice($search_results, $key, 0, [$promo]);
          break;
        }

      }

    }

    return $search_results;

  }

  /**
   * Helper function to retrieve promotions for specified search page.
   *
   * @param string $search_page_id
   *   ID of the custom search page being used.
   * @param string $search_keys
   *   The keys for which the search is being made.
   */
  private function retrievePromotions($search_page_id, $search_keys) {

    $query = $this->database->select('vais_promo', 'vp');
    $query->condition('vp.promo_search_page', $search_page_id, '=');
    $query->addField('vp', 'pid', 'pid');
    $query->addField('vp', 'promo_trigger', 'trigger');

    $results = $query->execute()->fetchAll();

    $promos = [];

    $searchKeysNormalized = strtolower(trim($search_keys));

    foreach ($results as $result) {

      $triggerWords = explode(',', $result->trigger);

      foreach ($triggerWords as $triggerWord) {
        $triggerWordNormalized = strtolower(trim($triggerWord));
        $promos[$triggerWordNormalized][] = $result->pid;
      }

    }

    // See if a promotion is available for the search keyword.
    if (empty($searchKeysNormalized) || empty($promos[$searchKeysNormalized])) {
      return [];
    }

    // Load the relevant promo.
    $vaisPromos = $this->eTypeManager->getStorage('vais_promo')->loadMultiple($promos[$searchKeysNormalized]);

    $promotions = [];

    /** @var /Drupal/vais_promos/Entity/VaisPromo $vaisPromo */
    foreach ($vaisPromos as $vaisPromo) {

      $promotion = [];

      // Basic promo information.
      $promotion['title'] = $vaisPromo->get('promo_title_override')->value;
      $promotion['description'] = $vaisPromo->get('promo_description')->value;
      $promotion['type'] = $vaisPromo->get('promo_type')->value;

      // Set the promo reference to the external link.
      $promotion['reference'] = $vaisPromo->get('promo_link')->value;

      // If the promotion uses referenced content, get the link.
      if ($promotion['type'] == 'internal') {

        // Get the id of the promotion referenced content.
        $promotion['content_id'] = $vaisPromo->get('promo_content')->target_id;

        // Check if the internal referenced content exists.
        /** @var \Drupal\node\Entity\Node|null $promoContent */
        $promoContent = $this->eTypeManager->getStorage('node')->load($promotion['content_id']);

        // If the referenced content does not exist, continue loop.
        if (empty($promoContent)) {
          continue;
        }

        // If the referenced content is not published, then continue loop.
        if ($promoContent->isPublished() === FALSE) {
          continue;
        }

        $promotion['reference'] = $this->aliasManager->getAliasByPath('/node/' . $promotion['content_id']);

      }

      $promotion['plugin_id'] = $this->getPluginId();

      $promotions[] = $promotion;

    }

    return $promotions;

  }

}
