<?php

namespace Drupal\vertex_ai_search\Plugin\rest\resource;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\search\Entity\SearchPage;
use Drupal\rest\ResourceResponse;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\vertex_ai_search\Service\VertexSearchManagerInterface;
use Drupal\vertex_ai_search\VertexSearchFilterPluginManager;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Represents search page results as resources.
 *
 * @RestResource(
 *   id = "vertex_ai_search_results",
 *   label = @Translation("Vertex AI Search Rest Resource"),
 *   class = "Drupal\vertex_ai_search_rest\Plugin\rest\resource\VertexSearchPageResource",
 *   provider = "restui",
 *   uri_paths = {
 *     "canonical" = "/search/vertex/v1/{search_page}"
 *   }
 * )
 */
class VertexAISearchResource extends ResourceBase {

  /**
   * The Vertex AI Search Manager.
   *
   * @var \Drupal\vertex_ai_search\Service\VertexSearchManagerInterface
   */
  protected $searchManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The link relation type manager used to create HTTP header links.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $linkRelationTypeManager;

  /**
   * Vertex Search Filter Plugin Manager.
   *
   * @var \Drupal\vertex_ai_search\VertexSearchFilterPluginManager
   */
  protected $filterPluginManager;

  /**
   * Constructs a Drupal\rest\Plugin\rest\resource\EntityResource object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\vertex_ai_search\Service\VertexSearchManagerInterface $search_manager
   *   The vertex search_manager service.
   * @param array $serializer_formats
   *   The available serialization formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $link_relation_type_manager
   *   The link relation type manager.
   * @param \Drupal\vertex_ai_search\VertexSearchFilterPluginManager $filterPluginManager
   *   Vertex Search Filter Plugin Manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    VertexSearchManagerInterface $search_manager,
    array $serializer_formats,
    LoggerInterface $logger,
    ConfigFactoryInterface $config_factory,
    PluginManagerInterface $link_relation_type_manager,
    VertexSearchFilterPluginManager $filterPluginManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->searchManager = $search_manager;
    $this->configFactory = $config_factory;
    $this->linkRelationTypeManager = $link_relation_type_manager;
    $this->filterPluginManager = $filterPluginManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('vertex_ai_search.search_manager'),
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('vertex_ai_search'),
      $container->get('config.factory'),
      $container->get('plugin.manager.link_relation_type'),
      $container->get('plugin.manager.vertex_search_filter')
    );
  }

  /**
   * Responds to entity GET requests.
   *
   * @param \Drupal\search\Entity\SearchPage $search_page
   *   The entity object, in this case a SearchPage.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The incoming request.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response containing the entity with its accessible fields.
   */
  public function get(SearchPage $search_page, Request $request) {

    $parameters['page'] = $request->query->get('page', 0);
    $parameters['keys'] = $request->query->get('keys');
    if (!empty($request->query->get('correction', NULL))) {
      $parameters['correction'] = $request->query->get('correction');
    }

    $searchPageConfig = $search_page->get('configuration');

    // Get path to search page on REST client.
    $parameters['search-path'] = $request->query->get(
      'search-path',
      $searchPageConfig['default_client_search_path']
    );

    $resource = $this->searchManager->executeSearch(
      $searchPageConfig,
      $parameters
    );

    // Wrap the resource in a ResourceResponse class and return.
    $resourceResponse = new ResourceResponse(['response' => $resource]);
    $resourceResponse->addCacheableDependency($parameters);

    return $resourceResponse;
  }

  /**
   * {@inheritdoc}
   */
  protected function getBaseRoute($canonical_path, $method) {

    $route = parent::getBaseRoute($canonical_path, $method);
    $parameters = $route->getOption('parameters') ?: [];
    $parameters['search_page']['type'] = 'entity:search_page';
    $route->setOption('parameters', $parameters);

    return $route;

  }

}
