<?php

namespace Drupal\Tests\vertex_ai_search\Kernel;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DependencyInjection\ServiceModifierInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\vertex_ai_search\Service\VertexSearchManager;

/**
 * Test to ensure 'vertex_ai_search.search_manager' service is reachable.
 *
 * @group vertex_ai_search
 *
 * @ingroup vertex_ai_search
 */
class VertexSearchManagerTest extends KernelTestBase implements ServiceModifierInterface {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['vertex_ai_search', 'search', 'token'];

  /**
   * The vertex_ai_search.search_manager service.
   *
   * @var \Drupal\vertex_ai_search\Service\VertexSearchManager
   */
  protected $searchManager;

  /**
   * {@inheritdoc}
   */
  public function alter(ContainerBuilder $container) {

    $service_definition = $container->getDefinition('vertex_ai_search.search_manager');

    // Override service class with subclass to make methods public for testing.
    $service_definition->setClass(TestVertexSearchManager::class);

  }

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->searchManager = $this->container->get('vertex_ai_search.search_manager');

  }

  /**
   * Test for existence of 'vertex_ai_search.search_manager' service.
   */
  public function testSearchManagerService() {

    $service = $this->container->get('vertex_ai_search.search_manager');
    $this->assertInstanceOf(VertexSearchManager::class, $service);

  }

  /**
   * Request Data Provider for Configuration and Parameters.
   */
  public function configRequestDataProvider() {

    // Each dataset has config, params, expected keyword, keyword validity.
    return [
      'noKeyword' => [
        [
          'google_cloud_project_id' => 'project_id',
          'google_cloud_location' => 'location',
          'vertex_ai_data_store_id' => 'data_store_id',
          'vertex_ai_serving_config' => 'serving_config',
          'exclusion_list' => NULL,
          'resultsPerPage' => 10,
          'safeSearch' => TRUE,
          'spelling_correction_mode' => 'AUTO',
        ],
        [
          'keys' => NULL,
          'correction' => NULL,
          'page' => 0,
        ],
        NULL,
        FALSE,
      ],
      'noKeywordAfterExclusion' => [
        [
          'google_cloud_project_id' => 'project_id',
          'google_cloud_location' => 'location',
          'vertex_ai_data_store_id' => 'data_store_id',
          'vertex_ai_serving_config' => 'serving_config',
          'exclusion_list' => 'testing',
          'resultsPerPage' => 10,
          'safeSearch' => TRUE,
          'spelling_correction_mode' => 'AUTO',
        ],
        [
          'keys' => 'testing',
          'correction' => NULL,
          'page' => 0,
        ],
        NULL,
        FALSE,
      ],
      'validValues1' => [
        [
          'google_cloud_project_id' => 'project_id',
          'google_cloud_location' => 'location',
          'vertex_ai_data_store_id' => 'data_store_id',
          'vertex_ai_serving_config' => 'serving_config',
          'exclusion_list' => 'testing',
          'resultsPerPage' => 10,
          'safeSearch' => TRUE,
          'spelling_correction_mode' => 'AUTO',
        ],
        [
          'keys' => 'testing search words',
          'correction' => NULL,
          'page' => 0,
        ],
        'search words',
        TRUE,
      ],
      'validValues2' => [
        [
          'google_cloud_project_id' => 'project_id',
          'google_cloud_location' => 'location',
          'vertex_ai_data_store_id' => 'data_store_id',
          'vertex_ai_serving_config' => 'serving_config',
          'exclusion_list' => '\d{3}-\d{3}',
          'resultsPerPage' => 10,
          'safeSearch' => TRUE,
          'spelling_correction_mode' => 'AUTO',
        ],
        [
          'keys' => '123-4567',
          'correction' => TRUE,
          'page' => 2,
        ],
        '7',
        TRUE,
      ],
    ];

  }

}

/**
 * A subclass for the vertex_ai_search.search_manager class.
 *
 * This allows changing the visibility of protected methods.
 * Methods must be public to be directly tested via phpUnit.
 */
class TestVertexSearchManager extends VertexSearchManager {

  /**
   * The Public version of prepareSearchRequest() in Vertex search_manager.
   *
   * @param array $search_page_config
   *   Configuration of the relevant custom search page.
   * @param array $search_parameters
   *   Array of search parameters (keys, page, etc...).
   *
   * @return \Google\Cloud\DiscoveryEngine\V1\SearchRequest
   *   A Vertex Search Request ready to be sent.
   */
  public function publicPrepareSearchRequest(array $search_page_config, array $search_parameters) {

    return $this->prepareSearchRequest($search_page_config, $search_parameters);

  }

}
