<?php

namespace Drupal\video_embed_dailymotion\Plugin\video_embed_field\Provider;

use Drupal\Core\Url;
use Drupal\video_embed_field\ProviderPluginBase;

/**
 * Dailymotion video provider plugin.
 *
 * @VideoEmbedProvider(
 *   id = "dailymotion",
 *   title = @Translation("Dailymotion")
 * )
 */
class Dailymotion extends ProviderPluginBase {

  /**
   * {@inheritdoc}
   */
  public function renderEmbedCode($width, $height, $autoplay, $title_format = NULL, $use_title_fallback = TRUE) {
    $embed_code = [
      '#type' => 'video_embed_iframe',
      '#provider' => 'dailymotion',
      '#url' => sprintf('//www.dailymotion.com/embed/video/%s', $this->getVideoId()),
      '#query' => [
        'autoPlay' => $autoplay,
      ],
      '#attributes' => [
        'width' => $width,
        'height' => $height,
        'frameborder' => '0',
        'allowfullscreen' => 'allowfullscreen',
      ],
    ];
    // We add a title attribute to the iframe tag for better accessibility.
    // This feature is new in Video Embed Field 3.0.x.
    // @see https://dequeuniversity.com/checklists/web/iframes
    $title = $this->getName($title_format, $use_title_fallback);
    if (isset($title)) {
      $embed_code['#attributes']['title'] = $title;
    }
    return $embed_code;
  }

  /**
   * {@inheritdoc}
   */
  public function getRemoteThumbnailUrl() {
    return sprintf('https://www.dailymotion.com/thumbnail/video/%s', $this->getVideoId());
  }

  /**
   * {@inheritdoc}
   */
  public static function getIdFromInput($input) {
    preg_match('/^https?:\/\/(www\.)?(dailymotion\.com|dai\.ly)\/(embed\/)?(video\/)?(?<id>[a-zA-Z0-9]*)(_([0-9a-zA-Z\-_])*)?$/', $input, $matches);
    return $matches['id'] ?? FALSE;
  }

  /**
   * Get the Youtube oembed data.
   *
   * @return array|null
   *   An array of data from the oembed endpoint or NULL if download failed.
   */
  protected function oEmbedData(): ?array {
    $normalized_url = sprintf('https://www.dailymotion.com/embed/video/%s', $this->videoId);
    $oembed_url = Url::fromUri('https://www.dailymotion.com/services/oembed', ['query' => ['url' => $normalized_url]]);
    return $this->downloadJsonData($oembed_url->toString());
  }

  /**
   * {@inheritdoc}
   */
  public function getName($title_format = NULL, $use_title_fallback = TRUE) {
    return $this->formatTitle(
      $this->oEmbedData()['title'] ?? NULL,
      $title_format,
      $use_title_fallback
    );
  }

}
