<?php

/**
 * @file
 * Contains \Drupal\video_embed_youku\Form\SettingsForm.
 */

namespace Drupal\video_embed_youku\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Video Embed Youku settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'video_embed_youku_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['video_embed_youku.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('video_embed_youku.settings');

    $form['api_client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Youku API Client ID'),
      '#description' => $this->t('Enter your Youku API Client ID. You can obtain this from the Youku Developer Portal.'),
      '#default_value' => $config->get('api_client_id'),
      '#required' => TRUE,
    ];

    $form['api_cache_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('API Cache Duration'),
      '#description' => $this->t('Duration in seconds to cache API responses. Default is 3600 (1 hour).'),
      '#default_value' => $config->get('api_cache_duration') ?: 3600,
      '#min' => 60,
      '#max' => 86400,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $client_id = $form_state->getValue('api_client_id');
    if (empty($client_id)) {
      $form_state->setErrorByName('api_client_id', $this->t('API Client ID is required.'));
    }

    $cache_duration = $form_state->getValue('api_cache_duration');
    if (!is_numeric($cache_duration) || $cache_duration < 60 || $cache_duration > 86400) {
      $form_state->setErrorByName('api_cache_duration', $this->t('Cache duration must be a number between 60 and 86400 seconds.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('video_embed_youku.settings')
      ->set('api_client_id', $form_state->getValue('api_client_id'))
      ->set('api_cache_duration', $form_state->getValue('api_cache_duration'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}