<?php

/**
 * @file
 * Contains \Drupal\video_embed_youku\Plugin\video_embed_field\Provider\Youku.
 */

namespace Drupal\video_embed_youku\Plugin\video_embed_field\Provider;

use Drupal\video_embed_field\ProviderPluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Youku video provider plugin.
 *
 * @VideoEmbedProvider(
 *   id = "youku",
 *   title = @Translation("Youku")
 * )
 */
class Youku extends ProviderPluginBase {
  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Constructs a new Youku provider instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, CacheBackendInterface $cache, LoggerChannelFactoryInterface $logger_factory, MessengerInterface $messenger, ClientInterface $http_client) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->cache = $cache;
    $this->logger = $logger_factory->get('video_embed_youku');
    $this->messenger = $messenger;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('cache.default'),
      $container->get('logger.factory'),
      $container->get('messenger'),
      $container->get('http_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function renderEmbedCode($width, $height, $autoplay) {
    $video_id = $this->getVideoId();
    if (!$video_id) {
      return [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => [
          'class' => ['video-embed-error'],
        ],
        '#value' => $this->t('Invalid Youku video ID.'),
      ];
    }

    return [
      '#type' => 'html_tag',
      '#tag' => 'iframe',
      '#attributes' => [
        'width' => $width,
        'height' => $height,
        'frameborder' => '0',
        'allowfullscreen' => 'allowfullscreen',
        'src' => sprintf('https://player.youku.com/embed/%s?autoplay=%d', $video_id, $autoplay),
      ],
    ];
  }

  /**
   * Get the Youku oEmbed data with caching.
   *
   * @return object|null
   *   An object of data from the Youku endpoint or NULL on failure.
   */
  protected function oEmbedData() {
    $video_id = $this->getVideoId();
    if (!$video_id) {
      return NULL;
    }

    $config = $this->configFactory->get('video_embed_youku.settings');
    $client_id = $config->get('api_client_id');
    $cache_duration = $config->get('api_cache_duration') ?: 3600;

    if (empty($client_id)) {
      $this->logger->warning('Youku API client ID is not configured.');
      $this->messenger->addWarning($this->t('Youku API client ID is not configured. Please configure it in the module settings.'));
      return NULL;
    }

    $cache_key = 'video_embed_youku:' . $video_id;
    $cached_data = $this->cache->get($cache_key);

    if ($cached_data && !empty($cached_data->data)) {
      return $cached_data->data;
    }

    try {
      $url = sprintf('https://api.youku.com/videos/show.json?client_id=%s&video_id=%s', $client_id, $video_id);
      $response = $this->httpClient->get($url, [
        'timeout' => 10,
        'headers' => [
          'User-Agent' => 'Drupal Video Embed Youku Module',
        ],
      ]);

      if ($response->getStatusCode() === 200) {
        $data = json_decode($response->getBody()->getContents());
        if ($data && !empty($data)) {
          $this->cache->set($cache_key, $data, time() + $cache_duration, ['video_embed_youku']);
          return $data;
        }
      }
    }
    catch (RequestException $e) {
      $this->logger->error('Failed to fetch Youku API data for video @video_id: @message', [
        '@video_id' => $video_id,
        '@message' => $e->getMessage(),
      ]);
      $this->messenger->addError($this->t('Failed to fetch video information from Youku. Please try again later.'));
    }
    catch (\Exception $e) {
      $this->logger->error('Unexpected error fetching Youku API data: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getRemoteThumbnailUrl() {
    $data = $this->oEmbedData();
    return $data && !empty($data->bigThumbnail) ? $data->bigThumbnail : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    $data = $this->oEmbedData();
    return $data && !empty($data->title) ? $data->title : $this->t('Youku Video');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    $data = $this->oEmbedData();
    return $data && !empty($data->description) ? $data->description : '';
  }

  /**
   * {@inheritdoc}
   */
  public static function getIdFromInput($input) {
    if (empty($input)) {
      return FALSE;
    }

    // Validate input is a string.
    if (!is_string($input)) {
      return FALSE;
    }

    $id = FALSE;
    
    try {
      // Parse_url is an easy way to break a url into its components.
      $parsed = parse_url($input);
      if (!isset($parsed['path'])) {
        return FALSE;
      }
      
      $path = $parsed['path'];
      $parts = explode('/', trim($path, '/'));
      
      foreach ($parts as $part) {
        if (strpos($part, 'id_') !== FALSE) {
          $id = str_replace('id_', '', $part);
          $id = str_replace('.html', '', $id);
          
          // Validate the ID format (Youku IDs are typically alphanumeric).
          if (preg_match('/^[a-zA-Z0-9]+$/', $id)) {
            return $id;
          }
        }
      }
    }
    catch (\Exception $e) {
      // Log error if needed, but return FALSE for invalid input.
    }

    return FALSE;
  }

}
