/**
 * @file
 * VideoJS player behavior for the player component.
 */
(function videojsMediablockIIFE(Drupal, once) {
  /**
   * Initialize VideoJS players with one-at-a-time playback behavior.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.videojsMediablockPlayer = {
    // Track all VideoJS players on the page
    allPlayers: [],

    /**
     * Pauses all other VideoJS players when one starts playing.
     *
     * @param {Object} currentPlayer - The current VideoJS player that is playing.
     */
    pauseOtherPlayers(currentPlayer) {
      this.allPlayers.forEach(function pausePlayerCallback(player) {
        // Only pause other players, not the one that's currently playing
        if (player !== currentPlayer && !player.paused()) {
          player.pause();
        }
      });
    },

    /**
     * Registers a VideoJS player in the global tracking list.
     * Also initializes the hotkeys plugin for the player.
     *
     * @param {Object} player - The VideoJS player instance to register.
     */
    registerPlayer(player) {
      if (
        player &&
        !player.el().hasAttribute('data-videojs-mediablock-processed')
      ) {
        // Add to tracked players list
        this.allPlayers.push(player);

        // Store 'this' for use in the event callback
        const self = this;

        // Add play event listener
        player.on('play', function playEventHandler() {
          self.pauseOtherPlayers(this);
        });

        // Initialize the hotkeys plugin for this player
        try {
          player.hotkeys({
            volumeStep: 0.1,
            seekStep: 5,
            enableModifiersForNumbers: false,
            enableVolumeScroll: false,
            enableHoverScroll: false,
          });
        } catch (e) {
          console.error('Error initializing videojs-hotkeys plugin:', e);
        }

        // Mark as processed
        player.el().setAttribute('data-videojs-mediablock-processed', 'true');
      }
    },

    attach(context) {
      // Reset the players list when the behavior is first attached
      if (context === document) {
        this.allPlayers = [];
      }

      // Find all VideoJS players in the context
      const videoElements = context.querySelectorAll('.video-js');
      const self = this;

      videoElements.forEach(function processVideoElement(element) {
        // Check if this is a VideoJS player and not already processed
        if (
          element.player &&
          !element.hasAttribute('data-videojs-mediablock-processed')
        ) {
          self.registerPlayer(element.player);
        }
        // Support for players that might initialize after this behavior runs
        else if (!element.hasAttribute('data-videojs-mediablock-processed')) {
          // Mark for processing
          element.setAttribute('data-videojs-mediablock-waiting', 'true');

          // Use MutationObserver to detect when videojs enhances this element
          const observer = new MutationObserver(
            function mutationObserverCallback() {
              if (
                element.player &&
                element.hasAttribute('data-videojs-mediablock-waiting')
              ) {
                self.registerPlayer(element.player);

                // Mark as processed and clean up
                element.removeAttribute('data-videojs-mediablock-waiting');

                // Disconnect observer once processed
                observer.disconnect();
              }
            },
          );

          // Observe the element for changes
          observer.observe(element, {
            attributes: true,
            childList: true,
            subtree: true,
          });
        }
      });
    },
  };

  /**
   * Register callback for when a new VideoJS player is ready.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.videojsMediablockPlayerEvents = {
    attach(context) {
      once('videojs-player-events', 'body', context).forEach(
        function setupPlayerEvents() {
          // Using native event system (no jQuery dependency)
          document.addEventListener(
            'videojs-player-ready',
            function playerReadyHandler(e) {
              try {
                const player = e.detail;
                if (player) {
                  Drupal.behaviors.videojsMediablockPlayer.registerPlayer(
                    player,
                  );
                }
              } catch (err) {
                console.error(
                  'Error handling videojs-player-ready event:',
                  err,
                );
              }
            },
          );
        },
      );
    },
  };

  /**
   * Helper to initialize VideoJS players with SDC components.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.videojsMediablockPlayerInit = {
    attach(context) {
      // Only initialize if videojs is available
      if (typeof window.videojs === 'undefined') {
        return;
      }

      once('videojs-elements', 'video.video-js', context).forEach(
        function initializeVideoElement(videoElement) {
          try {
            // Initialize VideoJS on the element
            window.videojs(videoElement, {}, function videojsReadyCallback() {
              // Player is ready
              this.el().setAttribute('data-videojs-initialized', 'true');

              // Trigger a native event (no jQuery dependency)
              try {
                const event = new CustomEvent('videojs-player-ready', {
                  detail: this,
                });
                document.dispatchEvent(event);
              } catch (err) {
                console.error(
                  'Error dispatching videojs-player-ready event:',
                  err,
                );
              }
            });
          } catch (e) {
            console.error('Error initializing VideoJS player:', e);
          }
        },
      );
    },
  };
})(Drupal, once);
