<?php

/**
 * @file
 * Form-related helper functions for VideoJS Mediablock module.
 */

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Component\Utility\UrlHelper;

/**
 * Helper function to initialize form storage.
 *
 * @param \Drupal\Core\Form\FormStateInterface $form_state
 *   The form state object.
 */
function _videojs_mediablock_initialize_form_storage(FormStateInterface &$form_state) {
  // Get storage values if they exist.
  $storage = $form_state->getStorage();
  if (!isset($storage['field_values'])) {
    $storage['field_values'] = [];
  }
  $form_state->setStorage($storage);
}

/**
 * Helper function to organize form fields.
 *
 * @param array $form
 *   The form array to modify.
 */
function _videojs_mediablock_organize_form_fields(array &$form) {
  // Create a fieldset to contain the media location and file.
  $form['media_hosting'] = [
    '#type' => 'fieldset',
    '#title' => t('Media hosting'),
    '#weight' => 0,
  ];

  // Associate the 'Media hosting' fieldset with fields, and set weights.
  $form['media_hosting']['field_videojs_media_location'] = $form['field_videojs_media_location'];
  $form['media_hosting']['field_videojs_media_location']['#weight'] = 0;
  $form['media_hosting']['field_videojs_local'] = $form['field_videojs_local'];
  $form['media_hosting']['field_videojs_local']['#weight'] = 1;
  $form['media_hosting']['field_videojs_remote'] = $form['field_videojs_remote'];
  $form['media_hosting']['field_videojs_remote']['#weight'] = 2;

  // Add media fields using helper function.
  $media_fields = _videojs_mediablock_get_media_fields();
  foreach ($media_fields as $field) {
    $form['media_hosting'][$field] = $form[$field];
  }

  // Unset original fields.
  unset($form['field_videojs_media_location']);
  unset($form['field_videojs_local']);
  unset($form['field_videojs_remote']);

  // Unset media fields.
  foreach ($media_fields as $field) {
    unset($form[$field]);
  }

  // Set up supporting media fieldset.
  $form['supporting_media'] = [
    '#type' => 'fieldset',
    '#title' => t('Optional media files and metadata'),
    '#weight' => 0,
  ];

  $form['supporting_media']['field_videojs_poster_image'] = $form['field_videojs_poster_image'];
  $form['supporting_media']['field_videojs_subtitle'] = $form['field_videojs_subtitle'];
  $form['supporting_media']['field_topics'] = $form['field_topics'];

  unset($form['field_videojs_poster_image']);
  unset($form['field_videojs_subtitle']);
  unset($form['field_topics']);

  // Without this next code, the form would render with a totally useless,
  // hierarchical UX/UI. This removes that.
  $form['supporting_media']['field_videojs_poster_image']['widget'][0]['#single_wrapped'] = 'skip';
  $form['supporting_media']['field_videojs_subtitle']['widget'][0]['#single_wrapped'] = 'skip';
  $form['supporting_media']['field_topics']['widget'][0]['#single_wrapped'] = 'skip';
}

/**
 * Helper function to apply permission restrictions.
 *
 * @param array $form
 *   The form array to modify.
 */
function _videojs_mediablock_apply_permissions(array &$form) {
  // Get current user.
  $current_user = \Drupal::currentUser();

  // Check permissions using helper function.
  $local_audio_access = _videojs_mediablock_check_field_permission('field_videojs_local_audio_file', $current_user);
  $local_video_access = _videojs_mediablock_check_field_permission('field_videojs_media_file', $current_user);
  $remote_audio_access = _videojs_mediablock_check_field_permission('field_videojs_remote_audio_file', $current_user);
  $remote_video_access = _videojs_mediablock_check_field_permission('field_videojs_remote_media_file', $current_user);
  $youtube_access = _videojs_mediablock_check_field_permission('field_videojs_youtube', $current_user);

  // Apply permission restrictions to specific fields.
  if (!$local_audio_access) {
    $form['media_hosting']['field_videojs_local_audio_file']['#access'] = FALSE;
  }
  if (!$local_video_access) {
    $form['media_hosting']['field_videojs_media_file']['#access'] = FALSE;
  }
  if (!$remote_audio_access) {
    $form['media_hosting']['field_videojs_remote_audio_file']['#access'] = FALSE;
  }
  if (!$remote_video_access) {
    $form['media_hosting']['field_videojs_remote_media_file']['#access'] = FALSE;
  }
  if (!$youtube_access) {
    $form['media_hosting']['field_videojs_youtube']['#access'] = FALSE;
  }

  // Modify options based on permissions
  // First, get the current options from the fields.
  if (isset($form['media_hosting']['field_videojs_local']['widget']['#options'])) {
    $local_options = $form['media_hosting']['field_videojs_local']['widget']['#options'];
    $local_has_options = FALSE;

    // Filter the options based on permissions.
    if (!$local_audio_access && isset($local_options['field_videojs_local_audio_file'])) {
      unset($form['media_hosting']['field_videojs_local']['widget']['#options']['field_videojs_local_audio_file']);
    }
    else {
      $local_has_options = TRUE;
    }

    if (!$local_video_access && isset($local_options['field_videojs_media_file'])) {
      unset($form['media_hosting']['field_videojs_local']['widget']['#options']['field_videojs_media_file']);
    }
    else {
      $local_has_options = TRUE;
    }

    // If no local options remain, disable the local option in the
    // media location field.
    if (!$local_has_options && isset($form['media_hosting']['field_videojs_media_location']['widget']['#options'])) {
      unset($form['media_hosting']['field_videojs_media_location']['widget']['#options']['field_videojs_local']);
    }
  }

  // Do the same for remote options.
  if (isset($form['media_hosting']['field_videojs_remote']['widget']['#options'])) {
    $remote_options = $form['media_hosting']['field_videojs_remote']['widget']['#options'];
    $remote_has_options = FALSE;

    // Filter the options based on permissions.
    if (!$remote_audio_access && isset($remote_options['field_videojs_remote_audio_file'])) {
      unset($form['media_hosting']['field_videojs_remote']['widget']['#options']['field_videojs_remote_audio_file']);
    }
    else {
      $remote_has_options = TRUE;
    }

    if (!$remote_video_access && isset($remote_options['field_videojs_remote_media_file'])) {
      unset($form['media_hosting']['field_videojs_remote']['widget']['#options']['field_videojs_remote_media_file']);
    }
    else {
      $remote_has_options = TRUE;
    }

    if (!$youtube_access && isset($remote_options['field_videojs_youtube'])) {
      unset($form['media_hosting']['field_videojs_remote']['widget']['#options']['field_videojs_youtube']);
    }
    else {
      $remote_has_options = TRUE;
    }

    // If no remote options remain, disable the remote option in
    // the media location field.
    if (!$remote_has_options && isset($form['media_hosting']['field_videojs_media_location']['widget']['#options'])) {
      unset($form['media_hosting']['field_videojs_media_location']['widget']['#options']['field_videojs_remote']);
    }
  }
}

/**
 * Helper function to add AJAX callbacks.
 *
 * @param array $form
 *   The form array to modify.
 */
function _videojs_mediablock_add_ajax_callbacks(array &$form) {
  // Add AJAX callbacks.
  $form['media_hosting']['field_videojs_media_location']['widget']['#ajax'] = [
    'callback' => 'videojs_mediablock_media_location_ajax_callback',
    'wrapper' => 'media-hosting-fields-wrapper',
    'event' => 'change',
  ];

  // Use the consolidated AJAX callback for both local and remote field types.
  $form['media_hosting']['field_videojs_local']['widget']['#ajax'] = [
    'callback' => 'videojs_mediablock_field_type_ajax_callback',
    'wrapper' => 'media-hosting-fields-wrapper',
    'event' => 'change',
  ];

  $form['media_hosting']['field_videojs_remote']['widget']['#ajax'] = [
    'callback' => 'videojs_mediablock_field_type_ajax_callback',
    'wrapper' => 'media-hosting-fields-wrapper',
    'event' => 'change',
  ];

  // Wrap all media hosting fields for AJAX refresh.
  $form['media_hosting']['#prefix'] = '<div id="media-hosting-fields-wrapper">';
  $form['media_hosting']['#suffix'] = '</div>';
}

/**
 * Helper function to set field visibility based on current values.
 *
 * @param array $form
 *   The form array to modify.
 * @param \Drupal\Core\Form\FormStateInterface $form_state
 *   The form state object.
 */
function _videojs_mediablock_set_field_visibility(array &$form, FormStateInterface &$form_state) {
  // Get current values using helper function.
  $entity = $form_state->getFormObject()->getEntity();
  $storage = $form_state->getStorage();

  // Media location - get simple value directly.
  $media_location = _videojs_mediablock_get_field_value(['field_videojs_media_location', 0, 'value'], $form_state, 'media_location', TRUE);

  // Local type - get simple value directly.
  $local_type = _videojs_mediablock_get_field_value(['field_videojs_local', 0, 'value'], $form_state, 'local_type', TRUE);

  // Remote type - get simple value directly.
  $remote_type = _videojs_mediablock_get_field_value(['field_videojs_remote', 0, 'value'], $form_state, 'remote_type', TRUE);

  // Store values in form state storage for AJAX rebuild persistence.
  if (!empty($media_location)) {
    $storage['media_location'] = $media_location;
  }
  if (!empty($local_type)) {
    $storage['local_type'] = $local_type;
  }
  if (!empty($remote_type)) {
    $storage['remote_type'] = $remote_type;
  }
  $form_state->setStorage($storage);

  // Prepopulate fields with stored values if we're rebuilding the form.
  if ($form_state->isRebuilding() && !empty($storage['field_values'])) {
    videojs_mediablock_prepopulate_fields($form, $form_state);
  }

  // Hide/show fields based on current values.
  $form['media_hosting']['field_videojs_local']['#access'] = ($media_location === 'field_videojs_local');
  $form['media_hosting']['field_videojs_remote']['#access'] = ($media_location === 'field_videojs_remote');

  $form['media_hosting']['field_videojs_local_audio_file']['#access'] = ($media_location === 'field_videojs_local' && $local_type === 'field_videojs_local_audio_file');
  $form['media_hosting']['field_videojs_media_file']['#access'] = ($media_location === 'field_videojs_local' && $local_type === 'field_videojs_media_file');

  $form['media_hosting']['field_videojs_remote_audio_file']['#access'] = ($media_location === 'field_videojs_remote' && $remote_type === 'field_videojs_remote_audio_file');
  $form['media_hosting']['field_videojs_remote_media_file']['#access'] = ($media_location === 'field_videojs_remote' && $remote_type === 'field_videojs_remote_media_file');
  $form['media_hosting']['field_videojs_youtube']['#access'] = ($media_location === 'field_videojs_remote' && $remote_type === 'field_videojs_youtube');

  // Make sure selection fields are properly set.
  if (!empty($media_location) && isset($form['media_hosting']['field_videojs_media_location'])) {
    $form['media_hosting']['field_videojs_media_location']['widget']['#default_value'] = $media_location;
  }

  if (!empty($local_type) && isset($form['media_hosting']['field_videojs_local'])) {
    $form['media_hosting']['field_videojs_local']['widget']['#default_value'] = $local_type;
  }

  if (!empty($remote_type) && isset($form['media_hosting']['field_videojs_remote'])) {
    $form['media_hosting']['field_videojs_remote']['widget']['#default_value'] = $remote_type;
  }

  // Check for entity values of selected field for validation and display.
  $entity_id = $entity->id();
  if ($entity_id) {
    // Determine the selected field using helper function.
    $selected_field = _videojs_mediablock_get_selected_field($media_location, $local_type, $remote_type);

    // If we have a selected field, check if the entity has a value for it.
    if ($selected_field && $entity->hasField($selected_field) && !$entity->get($selected_field)
      ->isEmpty()) {
      $entity_value = $entity->get($selected_field)->getValue();

      // Store the entity value in storage for backup.
      if (!isset($storage['field_values'])) {
        $storage['field_values'] = [];
      }
      $storage['field_values'][$selected_field] = $entity_value;
      $form_state->setStorage($storage);
    }
  }
}

/**
 * Helper function to get a field value from form state, storage, or entity.
 *
 * @param string|array $field_name
 *   The field name or an array path to the field value.
 * @param \Drupal\Core\Form\FormStateInterface $form_state
 *   The form state object.
 * @param string|null $storage_key
 *   Optional storage key to check in form state storage.
 * @param bool $simple_value
 *   Whether to return a simple value (TRUE) or the full field value (FALSE).
 *   For entity fields, simple value means the ->value property.
 *
 * @return mixed
 *   The field value, or NULL if not found.
 */
function _videojs_mediablock_get_field_value($field_name, FormStateInterface $form_state, $storage_key = NULL, $simple_value = FALSE) {
  // Try form state first.
  $value = $form_state->getValue($field_name);
  if (!empty($value)) {
    if ($simple_value && is_array($value) && isset($value[0]['value'])) {
      return $value[0]['value'];
    }
    return $value;
  }

  // Try storage next.
  $storage = $form_state->getStorage();
  if ($storage_key && !empty($storage[$storage_key])) {
    return $storage[$storage_key];
  }

  // Check if field_values exists and is an array before accessing it.
  if (isset($storage['field_values']) && is_array($storage['field_values'])) {
    // Extract storage key from array paths.
    $storage_field_name = $field_name;
    if (is_array($field_name) && !empty($field_name)) {
      $storage_field_name = $field_name[0];
    }

    // Use array_key_exists instead of !empty to avoid accessing the offset
    // directly.
    if (array_key_exists($storage_field_name, $storage['field_values'])) {
      $value = $storage['field_values'][$storage_field_name];
      if ($simple_value && is_array($value) && isset($value[0]['value'])) {
        return $value[0]['value'];
      }
      return $value;
    }
  }

  // Try entity last.
  $entity = $form_state->getFormObject()->getEntity();

  // If field_name is an array, extract the base field name for entity
  // operations.
  $entity_field_name = $field_name;
  if (is_array($field_name) && !empty($field_name)) {
    $entity_field_name = $field_name[0];
  }

  if ($entity && $entity->hasField($entity_field_name) && !$entity->get($entity_field_name)->isEmpty()) {
    if ($simple_value) {
      return $entity->get($entity_field_name)->value;
    }
    return $entity->get($entity_field_name)->getValue();
  }

  return NULL;
}

/**
 * Helper function for AJAX callbacks to handle common operations.
 *
 * This function centralizes common operations performed in AJAX callbacks,
 * reducing code duplication and improving maintainability.
 *
 * @param array $form
 *   The form array.
 * @param \Drupal\Core\Form\FormStateInterface $form_state
 *   The form state.
 * @param array $values
 *   Key-value pairs to store in form state storage.
 * @param array $options
 *   Additional options for handling field values:
 *   - field_value_key: The key for the field value to store.
 *   - field_value: The field value to store.
 *   - previous_value_key: The key for the previous field value to store.
 *   - previous_value: The previous field value to store.
 *   - entity: The entity to check for existing values.
 *
 * @return array
 *   The form element to return from the AJAX callback.
 */
function _videojs_mediablock_ajax_callback_helper(array &$form, FormStateInterface $form_state, array $values = [], array $options = []) {
  // Initialize storage.
  $storage = $form_state->getStorage();
  if (!isset($storage['field_values'])) {
    $storage['field_values'] = [];
  }

  // Store values in storage.
  foreach ($values as $key => $value) {
    $storage[$key] = $value;
  }

  // Store field value if provided.
  if (!empty($options['field_value_key']) && !empty($options['field_value'])) {
    $field_key = $options['field_value_key'];
    $field_value = $options['field_value'];

    // Store the field value in storage.
    $storage['field_values'][$field_key] = $field_value;
  }
  // If no field value provided but entity and field key are available, try to
  // retrieve from entity.
  elseif (!empty($options['field_value_key']) && !empty($options['entity'])) {
    $field_key = $options['field_value_key'];
    $entity = $options['entity'];

    if ($entity->id() && $entity->hasField($field_key) && !$entity->get($field_key)->isEmpty()) {
      $entity_value = $entity->get($field_key)->getValue();
      $storage['field_values'][$field_key] = $entity_value;
    }
  }

  // Store previous value if provided.
  if (!empty($options['previous_value_key']) &&
    !empty($options['previous_value']) &&
    $options['previous_value_key'] != $options['field_value_key']) {
    $prev_key = $options['previous_value_key'];
    $prev_value = $options['previous_value'];

    // Store the previous field value in storage.
    $storage['field_values'][$prev_key] = $prev_value;
  }

  // Set storage and rebuild form.
  $form_state->setStorage($storage);
  $form_state->setRebuild();

  // Return the appropriate form element.
  return $form['media_hosting'];
}

/**
 * Populates form fields from stored values during AJAX rebuilds.
 */
function videojs_mediablock_prepopulate_fields(array &$form, FormStateInterface $form_state) {
  $storage = $form_state->getStorage();

  // If we have stored field values, prepopulate them.
  if (!empty($storage['field_values'])) {
    foreach ($storage['field_values'] as $field_name => $field_value) {
      if (isset($form['media_hosting'][$field_name])) {
        // Extract value from a potentially nested field structure.
        $value = $field_value;

        // Set the default value for the field.
        $form['media_hosting'][$field_name]['widget']['#default_value'] = $value;
      }
    }
  }
}

/**
 * Helper function to validate a field value based on its type.
 *
 * @param string $field_name
 *   The field name to validate.
 * @param array $form
 *   The form array.
 * @param \Drupal\Core\Form\FormStateInterface $form_state
 *   The form state.
 *
 * @return bool
 *   TRUE if the field value is valid, FALSE otherwise.
 */
function _videojs_mediablock_validate_field_value($field_name, array $form, FormStateInterface &$form_state) {
  // Get storage values.
  $storage = $form_state->getStorage();

  // Get the entity to check if there's already a value.
  $entity = $form_state->getFormObject()->getEntity();

  // Try to get value from form state first.
  $value = $form_state->getValue($field_name);

  // If empty, check storage values.
  if (empty($value) && !empty($storage['field_values'][$field_name])) {
    $value = $storage['field_values'][$field_name];
  }

  // If still empty, check if entity already has a value (for existing content).
  if (empty($value) && $entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
    $value = $entity->get($field_name)->getValue();
  }

  // For remote fields, validate URL if value is provided.
  if (in_array($field_name, [
    'field_videojs_remote_audio_file',
    'field_videojs_remote_media_file',
    'field_videojs_youtube',
  ])) {
    // Check if we have a value and if it's empty.
    if (empty($value) || empty($value[0]['value'])) {
      $form_state->setErrorByName($field_name, t('Please provide a URL for the selected media field.'));
      return FALSE;
    }
    elseif (!UrlHelper::isValid($value[0]['value'], TRUE)) {
      $form_state->setErrorByName($field_name, t('Please provide a valid URL for the @field field.', [
        '@field' => $form['media_hosting'][$field_name]['#title'] ?? $field_name,
      ]));
      return FALSE;
    }
  }
  // For local fields, check if a file was selected.
  elseif (in_array($field_name, [
    'field_videojs_local_audio_file',
    'field_videojs_media_file',
  ])) {
    // Check if we have a value and if target_id is set.
    if (empty($value) || empty($value[0]['target_id'])) {
      // Verify this isn't an existing field value on an entity.
      $has_existing_value = FALSE;

      // Check entity directly first.
      if ($entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
        $existing_value = $entity->get($field_name)->getValue();
        $has_existing_value = !empty($existing_value[0]['target_id']);
      }

      // Also check if we have a valid value in storage.
      if (!$has_existing_value && !empty($storage['field_values'][$field_name])) {
        $stored_value = $storage['field_values'][$field_name];
        $has_existing_value = !empty($stored_value[0]['target_id']);
      }

      // Skip error if entity already has a value (for edit forms).
      if (!$has_existing_value) {
        // For extra safety, check if we're editing an existing entity.
        if ($entity->id()) {
          // One last attempt - try loading the entity fresh to check for
          // values.
          $entity_type_manager = \Drupal::entityTypeManager();
          $fresh_entity = $entity_type_manager->getStorage('block_content')
            ->load($entity->id());

          if ($fresh_entity && $fresh_entity->hasField($field_name) && !$fresh_entity->get($field_name)->isEmpty()) {
            $fresh_value = $fresh_entity->get($field_name)->getValue();
            $has_existing_value = !empty($fresh_value[0]['target_id']);

            // Store this value to ensure it's preserved.
            $storage['field_values'][$field_name] = $fresh_value;
            $form_state->setStorage($storage);
          }
        }

        // If we still have no value, set the error.
        if (!$has_existing_value) {
          $form_state->setErrorByName($field_name, t('Please select a file for the selected media field.'));
          return FALSE;
        }
      }
    }
  }

  return TRUE;
}

/**
 * Updates entity fields using form state values or storage values.
 *
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity to update.
 * @param \Drupal\Core\Form\FormStateInterface $form_state
 *   The form state.
 */
function videojs_mediablock_update_entity_from_storage(EntityInterface $entity, FormStateInterface $form_state) {
  // Use the helper function to process entity fields, but don't clear fields.
  _videojs_mediablock_process_entity_fields($entity, $form_state, [
    'clear_fields' => FALSE,
  ]);
}

/**
 * After build callback.
 */
function videojs_mediablock_after_build($form, FormStateInterface &$form_state) {
  // Get stored values.
  $storage = $form_state->getStorage();
  if (!isset($storage['field_values'])) {
    $storage['field_values'] = [];
    $form_state->setStorage($storage);
  }

  // Intentional placeholder for potential future use in AJAX rebuilds.
  // $entity = $form_state->getFormObject()->getEntity();
  return $form;
}
