<?php

/**
 * @file
 * Processing-related helper functions for VideoJS Mediablock module.
 */

use Drupal\Core\Entity\EntityInterface;

/**
 * Process entity fields for videojs mediablock entities.
 *
 * This helper function consolidates common operations performed on videojs
 * mediablock entities across different contexts. It reduces redundancy in
 * entity handling functions by centralizing the logic for determining the
 * selected field, checking permissions, clearing unused fields, and setting
 * field values.
 *
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity to process.
 * @param \Drupal\Core\Form\FormStateInterface|null $form_state
 *   The form state, if available.
 * @param array $options
 *   An array of options:
 *   - check_permissions: Whether to check permissions (default: FALSE).
 *   - clear_fields: Whether to clear unused fields (default: TRUE).
 *   - check_empty: Whether to check if fields are empty before clearing
 *     (default: FALSE).
 *   - context_check: Whether to perform context checks before clearing
 *     (default: FALSE).
 *   - build: The build array for rendering (only used when context_check
 *     is TRUE).
 *   - save_entity: Whether to save the entity after processing
 *     (default: FALSE). Can also be set to 'force' to save even if no
 *     fields were cleared.
 *   - queue_save: Whether to queue the entity for saving later
 *     (default: FALSE).
 *
 * @return array
 *   An array containing:
 *   - selected_field: The selected field name.
 *   - fields_cleared: Whether any fields were cleared.
 *   - entity: The processed entity.
 *   - build: The updated build array (if provided in options).
 */
function _videojs_mediablock_process_entity_fields(EntityInterface $entity, $form_state = NULL, $options = []) {
  // Set default options.
  $options += [
    'check_permissions' => FALSE,
    'clear_fields' => TRUE,
    'check_empty' => FALSE,
    'context_check' => FALSE,
    'build' => NULL,
    'save_entity' => FALSE,
    'queue_save' => FALSE,
  ];

  $result = [
    'selected_field' => NULL,
    'fields_cleared' => FALSE,
    'entity' => $entity,
    'build' => $options['build'],
  ];

  // Only process videojs_mediablock entities.
  if ($entity->getEntityTypeId() !== 'block_content' || $entity->bundle() !== 'videojs_mediablock') {
    return $result;
  }

  // Get field values from entity or form state.
  if ($form_state) {
    // Get values from form state.
    $media_location = _videojs_mediablock_get_field_value(['field_videojs_media_location', 0, 'value'], $form_state, 'media_location', TRUE);
    $local_type = _videojs_mediablock_get_field_value(['field_videojs_local', 0, 'value'], $form_state, 'local_type', TRUE);
    $remote_type = _videojs_mediablock_get_field_value(['field_videojs_remote', 0, 'value'], $form_state, 'remote_type', TRUE);
  }
  else {
    // Get values from entity.
    $field_values = _videojs_mediablock_get_entity_field_values($entity);
    $media_location = $field_values['media_location'];
    $local_type = $field_values['local_type'];
    $remote_type = $field_values['remote_type'];
  }

  // Determine which field should be kept.
  $selected_field = _videojs_mediablock_get_selected_field($media_location, $local_type, $remote_type);
  $result['selected_field'] = $selected_field;

  // Check permissions if requested.
  if ($options['check_permissions'] && !empty($selected_field)) {
    // Get user account ID from the entity owner if available.
    $owner_id = NULL;
    if ($entity->hasField('uid') && !$entity->get('uid')->isEmpty()) {
      $owner_id = $entity->get('uid')->target_id;
    }

    // Check permissions for the entity owner if available, otherwise use
    // current user.
    if ($owner_id) {
      $user_storage = \Drupal::entityTypeManager()->getStorage('user');
      $account = $user_storage->load($owner_id);
    }
    else {
      $account = \Drupal::currentUser();
    }

    // Check if the selected field is allowed for this user.
    $field_allowed = _videojs_mediablock_check_field_permission($selected_field, $account);

    if (!$field_allowed) {
      $entity->set($selected_field, NULL);
      // If the field is not allowed, don't preserve it when clearing other
      // fields.
      $selected_field = NULL;
      $result['selected_field'] = NULL;
    }
  }

  // Clear unused fields if requested.
  if ($options['clear_fields']) {
    $fields_to_check = _videojs_mediablock_get_media_fields();
    $fields_cleared = FALSE;

    // Process each field.
    foreach ($fields_to_check as $field) {
      if ($field !== $selected_field) {
        // Skip if field doesn't exist or is already empty.
        if (!$entity->hasField($field) ||
          ($options['check_empty'] && $entity->get($field)->isEmpty())) {
          continue;
        }

        // If context check is enabled, only clear fields in certain contexts.
        if ($options['context_check']) {
          // Don't render this field in the build array.
          if ($options['build'] && isset($options['build'][$field])) {
            $options['build'][$field]['#access'] = FALSE;
          }

          // Only modify the entity if we're in a form context or admin route.
          $is_admin_route = \Drupal::service('router.admin_context')->isAdminRoute();
          $is_form_op = FALSE;

          $route_match = \Drupal::routeMatch();
          $route_name = $route_match->getRouteName();

          if (strpos($route_name, 'entity.block_content.edit_form') === 0 ||
            strpos($route_name, 'block_content.add_form') === 0) {
            $is_form_op = TRUE;
          }

          if (!$is_admin_route && !$is_form_op) {
            continue;
          }
        }

        // Clear the field.
        $entity->set($field, NULL);
        $fields_cleared = TRUE;
      }
    }

    $result['fields_cleared'] = $fields_cleared;
  }

  // Set field values from form state if available.
  if ($form_state && $selected_field && $entity->hasField($selected_field)) {
    $storage = $form_state->getStorage();
    $value = $form_state->getValue($selected_field);

    // If the value is not in form state, try to get it from storage.
    if (empty($value) && isset($storage['field_values']) && !empty($storage['field_values'][$selected_field])) {
      $value = $storage['field_values'][$selected_field];
    }

    // If we still don't have a value but the entity already has one, keep it.
    if (empty($value) && !$entity->get($selected_field)->isEmpty()) {
      // Keep existing value (no-op read to satisfy linters).
      $entity->get($selected_field);
    }
    // Otherwise set the new value.
    elseif (!empty($value)) {
      $entity->set($selected_field, $value);
    }
  }

  // Save the entity if requested.
  if (($options['save_entity'] === 'force') ||
    ($options['save_entity'] && $result['fields_cleared'])) {
    $entity->save();
  }
  // Queue the entity for saving later if requested.
  elseif ($options['queue_save'] && $result['fields_cleared']) {
    drupal_register_shutdown_function('_videojs_mediablock_save_cleared_entity', $entity->id());
  }

  // Update the result with the processed entity and build array.
  $result['entity'] = $entity;
  $result['build'] = $options['build'];

  return $result;
}

/**
 * Saves an entity after its fields have been cleared.
 *
 * @param int $entity_id
 *   The entity ID to save.
 */
function _videojs_mediablock_save_cleared_entity($entity_id) {
  if (empty($entity_id)) {
    return;
  }

  try {
    $storage = \Drupal::entityTypeManager()->getStorage('block_content');
    $entity = $storage->load($entity_id);

    if ($entity) {
      // Use the helper function to process entity fields.
      _videojs_mediablock_process_entity_fields($entity, NULL, [
        'clear_fields' => TRUE,
        'check_empty' => TRUE,
        'save_entity' => 'force',
      ]);
    }
  }
  catch (\Exception $e) {
    \Drupal::logger('videojs_mediablock')
      ->error('Error saving cleared entity @id: @message', [
        '@id' => $entity_id,
        '@message' => $e->getMessage(),
      ]);
  }
}
