<?php

/**
 * @file
 * Selection-related helper functions for VideoJS Mediablock module.
 */

use Drupal\Core\Entity\EntityInterface;

/**
 * Determines which media field is currently selected based on configuration.
 *
 * This is a small, centralized helper used across preprocess and view logic to
 * avoid duplicating conditional checks for the media location and type fields.
 *
 * @param string $media_location
 *   Either 'field_videojs_local' or 'field_videojs_remote'. Any other value
 *   results in no selection.
 * @param string $local_type
 *   The selected local media field machine name when $media_location is local.
 * @param string $remote_type
 *   The selected remote media field machine name when $media_location is
 *   remote.
 *
 * @return string|null
 *   The selected media field machine name, or NULL if nothing is selected.
 */
function _videojs_mediablock_get_selected_field($media_location, $local_type, $remote_type) {
  if ($media_location === 'field_videojs_local' && !empty($local_type)) {
    return $local_type;
  }
  elseif ($media_location === 'field_videojs_remote' && !empty($remote_type)) {
    return $remote_type;
  }
  return NULL;
}

/**
 * Helper function to get media field values from an entity.
 *
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity to get field values from.
 *
 * @return array
 *   An array containing the media location, local type, and remote type values.
 */
function _videojs_mediablock_get_entity_field_values(EntityInterface $entity) {
  // Get the media location value.
  $media_location = '';
  if ($entity->hasField('field_videojs_media_location') && !$entity->get('field_videojs_media_location')->isEmpty()) {
    $media_location = $entity->get('field_videojs_media_location')->value;
  }

  // Get the local type value.
  $local_type = '';
  if ($entity->hasField('field_videojs_local') && !$entity->get('field_videojs_local')->isEmpty()) {
    $local_type = $entity->get('field_videojs_local')->value;
  }

  // Get the remote type value.
  $remote_type = '';
  if ($entity->hasField('field_videojs_remote') && !$entity->get('field_videojs_remote')->isEmpty()) {
    $remote_type = $entity->get('field_videojs_remote')->value;
  }

  return [
    'media_location' => $media_location,
    'local_type' => $local_type,
    'remote_type' => $remote_type,
  ];
}

/**
 * Returns the currently-selected media field for a given entity.
 *
 * This wraps _videojs_mediablock_get_entity_field_values() and
 * _videojs_mediablock_get_selected_field() to provide a single, readable
 * entry point for retrieving the active media field without duplicating
 * conditional logic. Does not modify the entity.
 *
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The videojs_mediablock block_content entity.
 *
 * @return string|null
 *   The selected media field machine name, or NULL if no selection exists.
 */
function _videojs_mediablock_get_selected_field_from_entity(EntityInterface $entity) {
  $field_values = _videojs_mediablock_get_entity_field_values($entity);
  return _videojs_mediablock_get_selected_field(
    $field_values['media_location'],
    $field_values['local_type'],
    $field_values['remote_type']
  );
}
