<?php

namespace Drupal\vidstack_player\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\media\Entity\MediaType;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\media\OEmbed\UrlResolverInterface;
use Drupal\media\Plugin\media\Source\OEmbedInterface;

/**
 * Plugin implementation of the Vidstack player remote video formatter.
 *
 * @FieldFormatter(
 *   id = "vidstack_player_remote_video",
 *   label = @Translation("Vidstack Player (Remote videos)"),
 *   field_types = {
 *     "link",
 *     "string",
 *     "string_long",
 *   }
 * )
 */
class VidstackPlayerRemoteVideoFormatter extends FormatterBase {
  use VidstackPlayerFormatterTrait;

  /**
   * The oEmbed url resolver service.
   *
   * @var \Drupal\media\OEmbed\UrlResolverInterface
   */
  protected $urlResolver;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs an VidstackPlayerRemoteVideoFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\media\OEmbed\UrlResolverInterface $urlResolver
   *   The oEmbed url resolver service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, UrlResolverInterface $urlResolver, EntityFieldManagerInterface $entityFieldManager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->urlResolver = $urlResolver;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('media.oembed.url_resolver'),
      $container->get('entity_field.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $summary['playback'] = $this->t('Playback controls: %controls', [
      '%controls' => $this->getSetting('controls') ? $this->t('visible') : $this->t('hidden'),
    ]);
    $summary['autoplay'] = $this->t('Autoplay: %autoplay', [
      '%autoplay' => $this->getSetting('autoplay') ? $this->t('yes') : $this->t('no'),
    ]);
    $summary['loop'] = $this->t('Loop: %loop', [
      '%loop' => $this->getSetting('loop') ? $this->t('yes') : $this->t('no'),
    ]);
    $summary += $this->getSharedSettingsSummary();
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    $settings = parent::defaultSettings();
    $settings['controls'] = FALSE;
    $settings['autoplay'] = FALSE;
    $settings['loop'] = FALSE;
    $settings += self::getSharedDefaultSettings();
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);
    $form['controls'] = [
      '#title' => $this->t('Show playback controls'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('controls'),
    ];
    $form['autoplay'] = [
      '#title' => $this->t('Autoplay'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('autoplay'),
    ];
    $form['loop'] = [
      '#title' => $this->t('Loop'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('loop'),
    ];
    $form += $this->getSharedSettings($form);
    return $form;
  }

  /**
   * Analyzes the URL string to make sure that a valid provider is used.
   *
   * @param string $url
   *   The video URL string.
   *
   * @return bool
   *   Returns whether or not a valid provider is used.
   *
   * @throws \Drupal\media\OEmbed\ProviderException
   * @throws \Drupal\media\OEmbed\ResourceException
   */
  private function usesValidProvider(string $url): bool {
    $provider_name = NULL;
    if ($provider = $this->urlResolver->getProviderByUrl($url)) {
      $provider_name = strtolower($provider->getName());
    }
    return in_array($provider_name, ['vimeo', 'youtube']);
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $settings = $this->getSettings();
    $themeAttributes = $this->buildThemeAttributes($settings, $items);

    foreach ($items as $delta => $item) {
      // Get the main property value (an url) and check if it uses a valid
      // provider:
      $main_property = $item->getFieldDefinition()->getFieldStorageDefinition()->getMainPropertyName();
      $value = $item->{$main_property};
      if (empty($value)) {
        continue;
      }
      if ($this->usesValidProvider($value)) {
        $elements[$delta] = [
          '#theme' => 'vidstack_player_remote_video'
        ];
        $elements[$delta]['#attributes'] = $themeAttributes;
        $elements[$delta]['#attributes']['src'] = $value;
      }
      else {
        $elements[$delta] = [
          '#type' => 'markup',
          '#markup' => $this->t('The video could not be loaded because @provider videos are not supported.', [
            '@provider' => $this->urlResolver->getProviderByUrl($value)->getName(),
          ]),
        ];
      }
    }
    if (!empty($elements)) {
      $elements['#attached']['library'][] = 'vidstack_player/init';
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    if ($field_definition->getTargetEntityTypeId() !== 'media') {
      return FALSE;
    }

    if (parent::isApplicable($field_definition)) {
      $media_type = $field_definition->getTargetBundle();

      if ($media_type) {
        $media_type = MediaType::load($media_type);
        return $media_type && $media_type->getSource() instanceof OEmbedInterface;
      }
    }
    return FALSE;
  }

}
