<?php

namespace Drupal\vidstack_player\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\file\Entity\File;
use Drupal\image\Entity\ImageStyle;
use Exception;

/**
 * Shared trait for Vidstack Player video formatters.
 *
 * NOTE, this trait does not portray ALL shared settings. Notably:
 * 'controls', 'autoplay' and 'loop' are shared, but not part of this trait.
 * This is because they get partially inherited by some formatters parent class.
 */
trait VidstackPlayerFormatterTrait {
  use StringTranslationTrait;

  /**
   * Returns the settings which are shared between formatters.
   *
   * @param array $form
   *   The form array.
   *
   * @return array
   *   The shared settings to reduce duplicate code.
   */
  protected function getSharedSettings(array $form) {
    // @see Trait class docblock.
    $form['muted'] = [
      '#title' => $this->t('Muted'),
      '#type' => 'checkbox',
      '#description' => $this->t('Start the video muted.'),
      '#default_value' => $this->getSetting('muted'),
    ];
    $form['load'] = [
      '#title' => $this->t('Load strategy'),
      '#type' => 'select',
      '#options' => $this->getLoadStrategyOptions(),
      '#description' => $this->t('The load strategy that will be used to determine when the video content should start loading.'),
      '#default_value' => $this->getSetting('load'),
    ];
    $form['poster'] = [
      '#title' => $this->t('Poster image'),
      '#type' => 'select',
      '#options' => [
        'none' => $this->t('None (Show first frame)'),
        'custom' => $this->t('Custom URL'),
        $this->t('Image fields from this entity:')->render() => $this->getEntityImageFields(),
      ],
      '#description' => $this->t('The image that will be displayed as a poster/thumbnail.'),
      '#default_value' => $this->getSetting('poster'),
    ];
    $form['posterUrl'] = [
      '#title' => $this->t('Custom poster image URL'),
      '#type' => 'textfield',
      '#description' => $this->t('The custom URL for the poster/thumbnail image.'),
      '#default_value' => $this->getSetting('posterUrl'),
      '#states' => [
        'visible' => [
          ':input[name*="[settings][poster]"]' => ['value' => 'custom'],
        ],
      ],
    ];
    $form['posterStyle'] = [
      '#title' => $this->t('Poster image style'),
      '#type' => 'select',
      '#options' => image_style_options(FALSE),
      '#empty_option' => $this->t('None (original image)'),
      '#description' => $this->t('The image style to use for the poster image.'),
      '#default_value' => $this->getSetting('posterStyle'),
    ];
    $form['posterLoad'] = [
      '#title' => $this->t('Poster load strategy'),
      '#type' => 'select',
      '#options' => [
        'eager' => $this->t('Load immediately'),
        'idle' => $this->t('Load after the page is loaded'),
        'visible' => $this->t('Load after becoming visible'),
      ],
      '#description' => $this->t('The load strategy that will be used to determine when the poster/thumbnail should start loading.'),
      '#default_value' => $this->getSetting('posterLoad'),
    ];
    return $form;
  }

  /**
   * Returns the load strategy form options.
   *
   * @return array
   *   The load strategy options.
   */
  protected function getLoadStrategyOptions() {
    return [
      'eager' => $this->t('Load immediately'),
      'idle' => $this->t('Load after the page is loaded'),
      'visible' => $this->t('Load after becoming visible'),
      'play' => $this->t('Load after hitting play'),
    ];
  }

  /**
   * Returns the default settings which are shared between formatters.
   *
   * @return array
   *   The shared default settings to reduce duplicate code.
   */
  protected static function getSharedDefaultSettings() {
    return [
      'muted' => FALSE,
      'load' => 'visible',
      'poster' => 'none',
      'posterUrl' => '',
      'posterLoad' => 'visible',
      'posterStyle' => '',
    ];
  }

  /**
   * Returns the settings summary shared between formatters.
   *
   * @return array
   *   The shared settings summary.
   */
  protected function getSharedSettingsSummary() {
    $summary = [];
    $summary['muted'] = $this->t('Muted: %muted', [
      '%muted' => $this->getSetting('muted') ? $this->t('yes') : $this->t('no'),
    ]);
    $summary['load'] = $this->t('Load strategy: %load', [
      '%load' => $this->getLoadStrategyOptions()[$this->getSetting('load')],
    ]);
    $posterValue = match($this->getSetting('poster')) {
      'none' => $this->t('None (Show first frame)'),
      'custom' => $this->t('Custom URL'),
      default => $this->t('Field "%name" (%style)', [
        '%name' => $this->getSetting('poster'),
        '%style' => $this->getSetting('posterStyle') === '' ? $this->t('None (original image)') : image_style_options(TRUE)[$this->getSetting('posterStyle')],
      ]),
    };
    $summary['poster'] = $this->t('Poster image: %poster', [
      '%poster' => $posterValue,
    ]);
    $summary['poster_load'] = $this->t('Poster load strategy: %load', [
      '%load' => $this->getLoadStrategyOptions()[$this->getSetting('posterLoad')],
    ]);
    return $summary;
  }

  /**
   * Get the poster url.
   */
  protected function getPosterUrl(FieldableEntityInterface $entity) {
    $settings = $this->getSettings();

    $posterUrl = '';
    switch ($settings['poster']) {
      case 'none':
        break;

      case 'custom':
        $posterUrl = $settings['posterUrl'];
        break;

      default:
        // Check if the poster is set to an image field on the entity, if not
        // break:
        if (!in_array($settings['poster'], array_keys($this->getEntityImageFields()), TRUE)) {
          break;
        }

        // Check if the poster field is empty (no referenced file). If not
        // break:
        $referenecedPosterFileId = $entity->get($settings['poster'])->target_id;
        if ($referenecedPosterFileId === NULL) {
          break;
        }

        // We have a proper file id now, load the file and generate the file
        // uri based on the "posterStyle" setting:
        $posterFile = File::load($referenecedPosterFileId);
        if ($settings['posterStyle'] === '') {
          $posterUrl = $posterFile->createFileUrl(FALSE);
        }
        else {
          $posterFileUri = $posterFile->getFileUri();
          $posterUrl = ImageStyle::load($settings['posterStyle'])->buildUrl($posterFileUri);
        }
        break;
    }
    return $posterUrl;
  }

  /**
   * Builds the theme attributes for the vidstack player element.
   */
  protected function buildThemeAttributes(array $settings, FieldableEntityInterface $entity) {
    $posterUrl = $this->getPosterUrl($entity);
    return [
      'class' => ['vidstack-player'],
      'data-autoplay' => (bool) $settings['autoplay'],
      'data-loop' => (bool) $settings['loop'],
      'data-muted' => (bool) $settings['muted'],
      'data-controls' => (bool) $settings['controls'],
      'data-load' => $settings['load'],
      'data-poster' => $posterUrl,
      'data-poster-load' => $settings['posterLoad'],
    ];
  }

  /**
   * Helper function that gathers all image fields attached to the media entity.
   *
   * @return array
   *   The image fields of the media entity.
   */
  public function getEntityImageFields(): array {
    $imageFieldsArray = [];
    $imageFields = array_filter($this->entityFieldManager->getFieldDefinitions($this->fieldDefinition->getTargetEntityTypeId(), $this->fieldDefinition->getTargetBundle()),
    function (FieldDefinitionInterface $field) {
      return $field->getType() === 'image';
    });
    foreach ($imageFields as $imageField) {
      $imageFieldsArray[$imageField->getName()] = $imageField->getLabel() . ' (' . $imageField->getName() . ')';
    }
    return $imageFieldsArray;
  }

  /**
   * Builds a render array that displays an error message.
   *
   * @param \Exception $e
   *   The error message to display.
   *
   * @return array
   *   The render array with the error message.
   */
  protected function buildError(Exception $e): array {
    return [
      '#theme' => 'vidstack_player_error',
      '#message' => $this->t('The video could not be loaded: %message', [
        '%message' => $e->getMessage(),
      ]),
      '#exception' => $e,
      '#type' => get_class($e),
    ];
  }

}
