<?php

namespace Drupal\vidstack_player\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\file\Plugin\Field\FieldFormatter\FileMediaFormatterBase;
use Drupal\file\Plugin\Field\FieldFormatter\FileMediaFormatterInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Template\Attribute;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the Vidstack player video formatter.
 *
 * @FieldFormatter(
 *   id = "vidstack_player_video",
 *   label = @Translation("Vidstack Player (Local videos)"),
 *   field_types = {
 *     "file"
 *   }
 * )
 */
class VidstackPlayerVideoFormatter extends FileMediaFormatterBase implements FileMediaFormatterInterface {
  use VidstackPlayerFormatterTrait;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs an VidstackPlayerRemoteVideoFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, EntityFieldManagerInterface $entityFieldManager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('entity_field.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function getMediaType() {
    return 'video';
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $summary += $this->getSharedSettingsSummary();

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    $settings = parent::defaultSettings();
    $settings += self::getSharedDefaultSettings();
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);
    $form += $this->getSharedSettings($form);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = parent::viewElements($items, $langcode);
    $settings = $this->getSettings();
    $themeAttributes = $this->buildThemeAttributes($settings, $items->getEntity());

    foreach ($items as $delta => $item) {
      // We want to keep the parent element, so we merge the arrays instead:
      $elements[$delta]['#theme'] = 'vidstack_player_video';
      $parentAttributes = $elements[$delta]['#attributes'];
      // If the parent attributes are not an array, we need to convert them
      // accordingly:
      if ($parentAttributes instanceof Attribute) {
        $parentAttributes = $parentAttributes->toArray();
      }
      elseif (!is_array($parentAttributes)) {
        $parentAttributes = [];
      }
      $elements[$delta]['#attributes'] = array_merge_recursive($parentAttributes, $themeAttributes);
    }
    $elements['#attached']['library'][] = 'vidstack_player/init';
    return $elements;
  }

}
