<?php

namespace Drupal\view_publish_toggle\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\node\NodeStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Handles AJAX requests to toggle the published state of a node.
 */
class PublishToggleController extends ControllerBase {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The node storage service.
   *
   * @var \Drupal\node\NodeStorageInterface
   */
  protected $nodeStorage;

  /**
   * Constructs a new PublishToggleController object.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   * @param \Drupal\node\NodeStorageInterface $node_storage
   *   The node storage service.
   */
  public function __construct(AccountProxyInterface $current_user, NodeStorageInterface $node_storage) {
    $this->currentUser = $current_user;
    $this->nodeStorage = $node_storage;
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpParamsInspection
   */
  public static function create(ContainerInterface $container) {
    return new static(
        $container->get('current_user'),
        $container->get('entity_type.manager')->getStorage('node')
    );
  }

  /**
   * AJAX callback to toggle the published state of a node.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response with the result.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *
   * @noinspection PhpPossiblePolymorphicInvocationInspection
   * @noinspection PhpParamsInspection
   */
  public function toggle(Request $request) {
    $nid = $request->request->get('nid');
    $node = $this->nodeStorage->load($nid);
    $user = $this->currentUser;

    if ($node && $node->access('update', $user) && $user->hasPermission('publish content toggle')) {

      $isPublished = $node->isPublished();
      if ($isPublished) {
        $node->setUnpublished();
      }
      else {
        $node->setPublished();
      }
      $node->save();

      return new JsonResponse([
        'status' => 'success',
        'published' => $node->isPublished(),
        'nid' => $nid,
      ]);
    }

    return new JsonResponse(['error' => 'Access denied'], 403);
  }

}
