<?php

namespace Drupal\view_publish_toggle\Plugin\Field\FieldFormatter;

use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a formatter to toggle the published state of a node.
 *
 * @FieldFormatter(
 *   id = "publish_toggle_formatter",
 *   label = @Translation("Publish toggle formatter"),
 *   field_types = {"boolean"}
 * )
 */
class PublishToggleFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * Constructs a PublishToggleFormatter object.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    AccountProxyInterface $current_user,
    CsrfTokenGenerator $csrf_token,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);

    $this->currentUser = $current_user;
    $this->csrfToken = $csrf_token;
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpParamsInspection
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
        $plugin_id,
        $plugin_definition,
        $configuration['field_definition'],
        $configuration['settings'] ?? [],
        $configuration['label'] ?? '',
        $configuration['view_mode'] ?? '',
        $configuration['third_party_settings'] ?? [],
        $container->get('current_user'),
        $container->get('csrf_token')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $node = $items->getEntity();

      if ($node->access('update', $this->currentUser) && $this->currentUser->hasPermission('publish content toggle')) {
        $published = (bool) $item->value;
        $status_class = $published ? 'marker--published' : 'marker--unpublished';
        $status_text = $published ? $this->t('Published') : $this->t('Unpublished');

        $elements[$delta] = [
          '#theme' => 'publish_toggle_formatter',
          '#nid' => $node->id(),
          '#published' => $published,
          '#status_class' => $status_class,
          '#status_text' => $status_text,
          '#attached' => [
            'library' => ['view_publish_toggle/publish-toggle'],
            'drupalSettings' => [
              'csrfToken' => $this->csrfToken->get('view_publish_toggle/ajax'),
            ],
          ],
        ];
      }
      else {
        $elements[$delta] = [
          '#markup' => $item->value ? $this->t('Published') : $this->t('Unpublished'),
        ];
      }
    }

    return $elements;
  }

}
