<?php

namespace Drupal\view_unpublished_dynamic_permission;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\BundlePermissionHandlerTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\PermissionHandlerInterface;

/**
 * Provides dynamic permissions viewing unpublished entities.
 */
class ViewUnpublishedDynamicPermission implements ContainerInjectionInterface {

  use AutowireTrait;
  use BundlePermissionHandlerTrait;
  use StringTranslationTrait;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected PermissionHandlerInterface $permissionHandler,
  ) {
  }

  /**
   * Returns an array of node type permissions.
   *
   * @return array
   *   The generated permissions.
   *
   * @see \Drupal\user\PermissionHandlerInterface::getPermissions()
   */
  public function generatePermissions(): array {
    $permissions = [];
    $manager = $this->entityTypeManager;

    foreach ($manager->getDefinitions() as $entity_type_id => $definition) {
      if (!is_a($definition->getClass(), EntityPublishedInterface::class, TRUE)) {
        continue;
      }

      // Check if entity is bundleable.
      if ($definition->hasKey('bundle') && $definition->getBundleEntityType()) {
        $bundle_type = $definition->getBundleEntityType();
        $bundles = $manager->getStorage($bundle_type)->loadMultiple();

        foreach ($bundles as $bundle_id => $bundle_entity) {
          $permissions["view any unpublished $entity_type_id:$bundle_id"] = [
            'title' => t('View any unpublished @bundle_label (@entity_label)', [
              '@bundle_label' => $bundle_entity->label(),
              '@entity_label' => $definition->getLabel(),
            ]),
          ];
        }
      }

      $permissions["view any unpublished $entity_type_id"] = [
        'title' => t('View any unpublished @label', [
          '@label' => $definition->getLabel(),
        ]),
      ];
    }

    return $permissions;
  }

}
