<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */
namespace Drupal\view_usernames\Plugin\EntityReferenceSelection;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\Attribute\EntityReferenceSelection;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\Plugin\EntityReferenceSelection\UserSelection as CoreUserSelection;
use Drupal\user\UserInterface;

/**
 * Ensures that ER fields do not leak usernames, even indirectly.
 *
 * Access checks already ensure that a user's name appears in the list only
 * when the current user has permission to view it. However, user IDs are still
 * exposed, and the Entity Reference Autocomplete widget's search feature can
 * be used to confirm whether a user exists.
 *
 * @internal This class is not part of the module's public programming API.
 */
#[EntityReferenceSelection(
  id: 'default:strict_user_filtered_by_view_usernames',
  label: new TranslatableMarkup('User selection filtered by View usernames'),
  group: 'default',
  weight: 128,
  entity_types: ['user']
)]
final class UserSelection extends CoreUserSelection {

  /**
   * Exception message for unsupported user creation operations.
   */
  private const UNSUPPORTED_OPERATION_MESSAGE = 'Creating a user through an entity reference field is not supported by Drupal core and has never been supported. See https://www.drupal.org/project/drupal/issues/2700411';

  /**
   * {@inheritdoc}
   */
  public function getReferenceableEntities($match = NULL, $match_operator = 'CONTAINS', $limit = 0): array {
    $options = ['user' => []];
    $storage = $this->entityTypeManager->getStorage('user');
    $start = 0;
    $remaining = $limit;

    do {
      $query = $this->buildEntityQuery($match, $match_operator);

      // When limit is 0, we want all results in one query (no range).
      // When limit > 0, fetch a batch that might contain inaccessible items.
      if ($limit > 0) {
        $query->range($start, $remaining);
      }

      $result = $query->execute();

      if (empty($result)) {
        break;
      }

      $start += count($result);

      foreach ($storage->loadMultiple($result) as $entity) {
        if ($entity->access('view label')) {
          $options['user'][$entity->id()] = Html::escape((string) ($this->entityRepository->getTranslationFromContext($entity)->label() ?? ''));

          if ($limit > 0) {
            $remaining--;
            if ($remaining <= 0) {
              return $options;
            }
          }
        }
      }

      // Continue only when we have a limit and need more items.
    } while ($limit > 0 && $remaining > 0);

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function countReferenceableEntities($match = NULL, $match_operator = 'CONTAINS'): int {
    // Highly inefficient, but this is the best we can do since view username
    // deciders cannot filter at the query level.
    return count($this->getReferenceableEntities($match, $match_operator)['user']);
  }

  /**
   * {@inheritdoc}
   */
  public function validateReferenceableEntities(array $ids) {
    $valid_ids = [];
    foreach ($this->entityTypeManager->getStorage('user')->loadMultiple($ids) as $user) {
      if ($user->access('view label')) {
        $valid_ids[] = $user->id();
      }
    }
    return $valid_ids;
  }

  /**
   * {@inheritdoc}
   */
  public function validateReferenceableNewEntities(array $entities): array {
    throw new \LogicException(self::UNSUPPORTED_OPERATION_MESSAGE);
  }

  /**
   * {@inheritdoc}
   */
  public function createNewEntity($entity_type_id, $bundle, $label, $uid): UserInterface {
    throw new \LogicException(self::UNSUPPORTED_OPERATION_MESSAGE);
  }

}
