<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */
namespace Drupal\Tests\view_usernames\Kernel;

use Drupal\Core\Access\AccessResultAllowed;
use Drupal\Core\Access\AccessResultForbidden;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Session\AccountInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\user\UserInterface;
use Drupal\view_usernames\Contracts\ViewUsernameAccessDeciderInterface;
use Drupal\view_usernames\Plugin\EntityReferenceSelection\UserSelection;
use PHPUnit\Framework\Attributes\CoversClass;

/**
 * Tests the UserSelection plugin for entity reference fields.
 *
 * @internal This class is not part of the module's public programming API.
 */
#[CoversClass(UserSelection::class)]
final class UserSelectionTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'view_usernames',
  ];

  /**
   * The closure that determines username view access.
   *
   * @var \Closure(\Drupal\Core\Session\AccountInterface, \Drupal\user\UserInterface): (\Drupal\Core\Access\AccessResultAllowed|\Drupal\Core\Access\AccessResultForbidden)
   */
  protected \Closure $decider;

  /**
   * The user selection plugin instance being tested.
   *
   * @var \Drupal\view_usernames\Plugin\EntityReferenceSelection\UserSelection
   */
  private UserSelection $userSelection;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('user');
    $this->installConfig(['user']);

    // This is already an assertion on the type of the object returned
    // by the plugin manager.
    $this->userSelection = $this->container->get('plugin.manager.entity_reference_selection')->getInstance([
      'target_type' => 'user',
      'handler' => 'default',
      'target_bundles' => 'user',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function register(ContainerBuilder $container): void {
    parent::register($container);

    $decider = function (AccountInterface $acting_user, UserInterface $other_user): AccessResultAllowed|AccessResultForbidden {
      return ($this->decider)($acting_user, $other_user);
    };

    $container->set('view_usernames.view_username_access_decider', new ConfigurableViewUserNameDeciderMock($decider));
  }

  /**
   * Tests that getReferenceableEntities invokes the view username decider.
   */
  public function testGetReferenceableEntitiesInvokesViewUsernameDeciderWhenCheckingViewLabelAccess(): void {
    $this->createUser();
    $was_decider_called = FALSE;
    $this->decider = static function () use (&$was_decider_called): AccessResultAllowed {
      $was_decider_called = TRUE;
      return AccessResultAllowed::allowed();
    };

    $this->userSelection->getReferenceableEntities();

    self::assertTrue($was_decider_called, 'The ER selection plugin for the user entity performed a "view label" entity access check which made a decision based on the view username decider.');
  }

  /**
   * Tests that getReferenceableEntities respects limit with mixed access.
   */
  public function testGetReferenceableEntitiesReturnsExactLimitOfAllowedEntitiesWhenAccessDeniedBetweenItems(): void {
    $this->createUser(name: 'tarzan1');
    $this->createUser(name: 'jane1');
    // Ensure auto-pagination does not rely on entity id increments.
    $this->createUser(name: 'tarzan2')->delete();
    $this->createUser(name: 'jane2')->delete();
    $this->createUser(name: 'tarzan3');
    $this->createUser(name: 'jane3');

    $this->decider = static function (AccountInterface $acting_user, UserInterface $other_user): AccessResultAllowed|AccessResultForbidden {
      if (str_contains($other_user->getAccountName(), 'tarzan')) {
        return AccessResultAllowed::allowed();
      }
      return AccessResultForbidden::forbidden();
    };

    $items = $this->userSelection->getReferenceableEntities(limit: 2)['user'];

    self::assertCount(2, $items, 'Exactly 2 entities returned respecting the limit.');
    self::assertEqualsCanonicalizing(['tarzan1', 'tarzan3'], $items, 'The view access decider only allowed returning Tarzans but not Janes.');
  }

  /**
   * Tests that getReferenceableEntities stops pagination when limit reached.
   */
  public function testGetReferenceableEntitiesStopsAutoPaginationWhenLimitReached(): void {
    $this->createUser();
    $this->createUser();
    $this->createUser();
    $this->createUser();
    $this->createUser();

    $decider_call_counter = 0;
    $this->decider = static function () use (&$decider_call_counter): AccessResultAllowed {
      $decider_call_counter++;
      return AccessResultAllowed::allowed();
    };

    $this->userSelection->getReferenceableEntities(limit: 2);

    self::assertEquals(2, $decider_call_counter, 'Auto-pagination stopped after the necessary amount of entities could be collected.');
  }

  /**
   * Tests that getReferenceableEntities returns all users when limit is zero.
   */
  public function testGetReferenceableEntitiesReturnsAllAccessibleUsersWhenLimitIsZero(): void {
    $this->createUser(name: 'tarzan1');
    $this->createUser(name: 'jane1');
    $this->createUser(name: 'tarzan2');
    $this->createUser(name: 'jane2');
    $this->createUser(name: 'tarzan3');

    $this->decider = static function (AccountInterface $acting_user, UserInterface $other_user): AccessResultAllowed|AccessResultForbidden {
      if (str_contains($other_user->getAccountName(), 'tarzan')) {
        return AccessResultAllowed::allowed();
      }
      return AccessResultForbidden::forbidden();
    };

    $items = $this->userSelection->getReferenceableEntities(limit: 0)['user'];

    self::assertCount(3, $items, 'All three Tarzan users were returned with unlimited limit.');
    self::assertEqualsCanonicalizing(['tarzan1', 'tarzan2', 'tarzan3'], $items, 'Only accessible users were included.');
  }

  /**
   * Tests that getReferenceableEntities returns all users when no limit set.
   */
  public function testGetReferenceableEntitiesReturnsAllAccessibleUsersWhenNoLimitSpecified(): void {
    $this->createUser(name: 'user1');
    $this->createUser(name: 'user2');
    $this->createUser(name: 'user3');

    $this->decider = static function (): AccessResultAllowed {
      return AccessResultAllowed::allowed();
    };

    $items = $this->userSelection->getReferenceableEntities()['user'];

    self::assertCount(3, $items, 'All users returned when no limit specified (defaults to 0).');
  }

  /**
   * Tests that getReferenceableEntities returns empty when no accessible users.
   */
  public function testGetReferenceableEntitiesReturnsEmptyArrayWhenNoAccessibleUsers(): void {
    $this->createUser(name: 'user1');
    $this->createUser(name: 'user2');

    $this->decider = static function (): AccessResultForbidden {
      return AccessResultForbidden::forbidden();
    };

    $items = $this->userSelection->getReferenceableEntities()['user'];

    self::assertEmpty($items, 'No users returned when all access is denied.');
  }

  /**
   * Tests that countReferenceableEntities only counts accessible entities.
   */
  public function testCountReferenceableEntitiesCountsOnlyAccessibleEntities(): void {
    $this->createUser(name: 'tarzan1');
    $this->createUser(name: 'jane1');
    $this->createUser(name: 'tarzan2');
    $this->createUser(name: 'jane2');

    $this->decider = static function (AccountInterface $acting_user, UserInterface $other_user): AccessResultAllowed|AccessResultForbidden {
      if (str_contains($other_user->getAccountName(), 'tarzan')) {
        return AccessResultAllowed::allowed();
      }
      return AccessResultForbidden::forbidden();
    };

    self::assertEquals(2, $this->userSelection->countReferenceableEntities(), 'Only allowed entities are counted.');
  }

  /**
   * Tests that validateReferenceableEntities returns only accessible user IDs.
   */
  public function testValidateReferenceableEntitiesReturnsOnlyAccessibleUserIds(): void {
    $expected = [];
    $tarzanIds = [];
    $janeIds = [];
    $expected[] = $tarzanIds[] = $this->createUser(name: 'tarzan1')->id();
    $janeIds[] = $this->createUser(name: 'jane1')->id();
    $tarzan2 = $this->createUser(name: 'tarzan2');
    $tarzanIds[] = $tarzan2->id();
    $tarzan2->delete();
    $jane2 = $this->createUser(name: 'jane2');
    $janeIds[] = $jane2->id();
    $jane2->delete();
    $expected[] = $tarzanIds[] = $this->createUser(name: 'tarzan3')->id();
    $janeIds[] = $this->createUser(name: 'jane3')->id();

    $this->decider = static function (AccountInterface $acting_user, UserInterface $other_user): AccessResultAllowed|AccessResultForbidden {
      if (str_contains($other_user->getAccountName(), 'tarzan')) {
        return AccessResultAllowed::allowed();
      }
      return AccessResultForbidden::forbidden();
    };

    $items = $this->userSelection->validateReferenceableEntities(array_merge($tarzanIds, $janeIds));

    self::assertEqualsCanonicalizing($expected, $items, "Only the two existing Tarzans' id was returned.");
  }

  /**
   * Tests that validateReferenceableNewEntities throws LogicException.
   */
  public function testValidateReferenceableNewEntitiesThrowsLogicException(): void {
    $this->expectException(\LogicException::class);
    $this->expectExceptionMessage('Creating a user through an entity reference field is not supported by Drupal core and has never been supported. See https://www.drupal.org/project/drupal/issues/2700411');

    $this->userSelection->validateReferenceableNewEntities([]);
  }

  /**
   * Tests that createNewEntity throws LogicException.
   */
  public function testCreateNewEntityThrowsLogicException(): void {
    $this->expectException(\LogicException::class);
    $this->expectExceptionMessage('Creating a user through an entity reference field is not supported by Drupal core and has never been supported. See https://www.drupal.org/project/drupal/issues/2700411');

    $this->userSelection->createNewEntity('user', 'user', 'test_user', 1);
  }

}

/**
 * Mock implementation of ViewUsernameAccessDeciderInterface for testing.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class ConfigurableViewUserNameDeciderMock implements ViewUsernameAccessDeciderInterface {

  /**
   * Constructs a new object.
   *
   * @param \Closure(\Drupal\Core\Session\AccountInterface, \Drupal\user\UserInterface): (\Drupal\Core\Access\AccessResultAllowed|\Drupal\Core\Access\AccessResultForbidden) $decider
   *   The closure that performs the access decision.
   */
  public function __construct(private readonly \Closure $decider) {}

  /**
   * {@inheritdoc}
   */
  public function canViewUserName(AccountInterface $acting_user, UserInterface $other_user): AccessResultAllowed|AccessResultForbidden {
    return ($this->decider)($acting_user, $other_user);
  }

}
