<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2023 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

namespace Drupal\view_usernames_node_author\Cache\Context;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\Context\CalculatedCacheContextInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\node\Cache\NodeAccessGrantsCacheContext;
use Drupal\user\UserStorageInterface;

/**
 * Defines a cache context "per acting user's node grants" caching.
 *
 * Cache context ID: 'node_grants_by_acting_user:||%uid' (to vary by all
 * operations' grants).
 * Calculated cache context ID: 'node_grants_by_acting_user:%operation||%uid',
 * e.g., 'user.node_grants:view||%uid' (to vary by the view operation's grants).
 *
 * The built-in NodeAccessGrantsCacheContext always calculates grants for the
 * current user that might not be the same as the acting user. This cache
 * context SHOULD BE only used in that case otherwise it is redundant.
 *
 * @internal This class is not part of the module's public programming API.
 *   Cache context id may also change in the future.
 */
final class NodeGrantsByActingUserCacheContext implements CalculatedCacheContextInterface {

  private const OPERATION_USERNAME_SEPARATOR = '||';

  private const CACHE_CONTEXT_ID = 'node_grants_by_acting_user';

  /**
   * The user storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  private UserStorageInterface $userStorage;

  /**
   * Constructs a new object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->userStorage = $entity_type_manager->getStorage('user');
  }

  /**
   * {@inheritdoc}
   */
  public static function getLabel(): TranslatableMarkup {
    return new TranslatableMarkup('Node grants by acting user');
  }

  /**
   * {@inheritdoc}
   */
  public function getContext($parameter = NULL): string {
    [$operation] = explode(self::OPERATION_USERNAME_SEPARATOR, $parameter ?? '');

    return $this->buildNodeGrantsCacheContextForParameters($parameter)->getContext($operation);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata($parameter = NULL): CacheableMetadata {
    [$operation] = explode(self::OPERATION_USERNAME_SEPARATOR, $parameter ?? '');

    return $this->buildNodeGrantsCacheContextForParameters($parameter)->getCacheableMetadata($operation);
  }

  /**
   * Builds a node grants cache context object.
   *
   * @param string|null $parameter
   *   The incoming context parameter.
   *
   * @return \Drupal\node\Cache\NodeAccessGrantsCacheContext
   *   The built cache context based on the input parameters.
   *
   * @todo Should we keep an internal object cache for performance reasons?
   */
  private function buildNodeGrantsCacheContextForParameters(?string $parameter): NodeAccessGrantsCacheContext {
    [, $acting_user_id] = explode(self::OPERATION_USERNAME_SEPARATOR, $parameter ?? '')
    + ['', NULL];
    if ($acting_user_id === NULL) {
      throw new \LogicException('The acting user id cannot be null.');
    }

    /** @var \Drupal\user\UserInterface|null $acting_user */
    $acting_user = $this->userStorage->load($acting_user_id);
    if ($acting_user) {
      return new NodeAccessGrantsCacheContext($acting_user);
    }

    // Calculate node grants for the anonymous user as a safe fallback.
    // @phpstan-ignore argument.type
    return new NodeAccessGrantsCacheContext($this->userStorage->load(0));
  }

  /**
   * Generates a cache context id from a user.node_grants cache context id.
   *
   * @param string $user_node_grants_context_id
   *   A user.node_grants cache context id string.
   * @param int $acting_user_id
   *   The UID of the acting user.
   *
   * @return string
   *   The cache context if for this cache context.
   */
  public static function generateContextIdFromUserNodeGrantsContextId(string $user_node_grants_context_id, int $acting_user_id): string {
    if (str_starts_with($user_node_grants_context_id, 'user.node_grants')) {
      [, $operation] = explode(':', $user_node_grants_context_id) + [NULL, NULL];
      if ($operation === NULL) {
        return sprintf('%s:%s%s', self::CACHE_CONTEXT_ID, self::OPERATION_USERNAME_SEPARATOR, $acting_user_id);
      }

      return sprintf('%s:%s%s%s', self:: CACHE_CONTEXT_ID, $operation, self::OPERATION_USERNAME_SEPARATOR, $acting_user_id);
    }

    throw new \LogicException(sprintf('"%s" is not a user node grants cache context.', $user_node_grants_context_id));
  }

}
