<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2023 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

namespace Drupal\view_usernames_node_author;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultAllowed;
use Drupal\Core\Access\AccessResultForbidden;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeStorageInterface;
use Drupal\user\UserInterface;
use Drupal\view_usernames\Contracts\ViewUsernameAccessDeciderInterface;
use Drupal\view_usernames_node_author\Cache\Context\NodeGrantsByActingUserCacheContext;

/**
 * Grants access to node author's username.
 *
 * The access is only granted if at least one node that the acting user has
 * access and authored by the other user.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class NodeAuthorViewUsernameAccessDecider implements ViewUsernameAccessDeciderInterface {

  private const SUPPORTED_ENTITY_OPERATIONS = ['view', 'update', 'delete'];

  /**
   * The node storage.
   *
   * @var \Drupal\node\NodeStorageInterface
   */
  private NodeStorageInterface $nodeStorage;

  /**
   * Node definition.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface
   */
  private EntityTypeInterface $nodeDefinition;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  private RendererInterface $renderer;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  private AccountInterface $currentUser;

  /**
   * The entity operation access to be checked.
   *
   * @var string
   */
  private string $entityOperation = 'view';

  /**
   * The number of entities to be loaded at once to verify entity access.
   *
   * @var int<1,max>
   */
  private int $entityLoadBatchSize = 20;

  /**
   * Constructs a new object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, RendererInterface $renderer, AccountInterface $current_user) {
    $this->nodeStorage = $entity_type_manager->getStorage('node');
    // @phpstan-ignore assign.propertyType
    $this->nodeDefinition = $entity_type_manager->getDefinition('node');
    $this->renderer = $renderer;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public function canViewUserName(AccountInterface $acting_user, UserInterface $other_user): AccessResultAllowed|AccessResultForbidden {
    $cacheability = CacheableMetadata::createFromRenderArray([]);

    // When access checking happens node_query_node_access_alter() is called
    // that tries to bubble up cacheability information with early rendering.
    $authored_node_ids = $this->executeCallableInRenderContext(fn() => $this->getAuthoredNodesByUser($other_user, $acting_user, TRUE), $cacheability);

    // The bad news is that hook_node_access() can override hook_node_grants()
    // and grant access to a node that hook_node_grants() denied access in
    // query time.
    // @see \Drupal\Tests\view_usernames_node_author\Kernel\NodeAccessTest::testHookNodeAccessCanOverrideAccessGrantedByHookNodeGrants()
    if ($authored_node_ids === []) {
      $authored_node_ids = $this->getAuthoredNodesByUser($other_user, $acting_user, FALSE);
    }

    if ($authored_node_ids !== []) {
      // Do a memory friendly iteration on node objects and bail out as early
      // as possible.
      foreach (array_chunk($authored_node_ids, $this->entityLoadBatchSize) as $id_chunks) {
        foreach ($this->nodeStorage->loadMultiple($id_chunks) as $node) {
          // The bad news is that hook_node_access() can override
          // hook_node_grants() and deny access to a node that
          // hook_node_grants() granted access in query time.
          // @see \Drupal\Tests\view_usernames_node_author\Kernel\NodeAccessTest::testHookNodeAccessGrantsCanNotOverrideAccessGrantedByHookNodeAccess()
          $node_access = $node->access($this->entityOperation, $acting_user, TRUE);
          assert($node_access instanceof AccessResultInterface);
          if ($node_access->isAllowed()) {
            if ($node_access instanceof CacheableDependencyInterface) {
              return AccessResult::allowed()->addCacheableDependency($this->fixCacheability($acting_user, $node_access));
            }

            assert($node_access instanceof AccessResultAllowed);
            return $node_access;
          }

          // We must bubble up cacheability information here because we cannot
          // know what invalidates the previous not allowed result.
          $cacheability->addCacheableDependency($node_access);
        }
      }
    }

    // When a new node is created or updated, invalidate cached result because
    // that may change the previously calculated answer here.
    $cacheability->addCacheTags($this->nodeDefinition->getListCacheTags());

    $cacheability = $this->fixCacheability($acting_user, $cacheability);

    return AccessResult::forbidden()->addCacheableDependency($cacheability);
  }

  /**
   * Fixes cacheability information calculated by Drupal core.
   *
   * The entity access system in Drupal core always sets dependencies on the
   * currently logged-in user instead of the "acting user" in entity access
   * checks. (The $account parameter in method calls.)
   * In this case this is especially a problem because there is no other way
   * to ensure proper cache invalidation than making sure the generated render
   * result varies per an acting user's node grants and not by the current
   * user's node grants. (Reminder, a user's access to a node can change
   * without a user- or a node entity update with hook_node_grants(), therefore
   * adding user- and node related cache tags to a result is not enough to
   * ensure proper cache invalidation.)
   *
   * @param \Drupal\Core\Session\AccountInterface $acting_user
   *   The currently active user.
   * @param \Drupal\Core\Cache\CacheableDependencyInterface $cacheability
   *   The cacheability information to be checked and fixed if necessary.
   *
   * @return \Drupal\Core\Cache\CacheableDependencyInterface
   *   The correct cacheability metadata.
   *
   * @see \node_query_node_access_alter()
   * @see \Drupal\node\NodeGrantDatabaseStorage::access()
   * @see https://www.drupal.org/project/drupal/issues/3340246
   */
  private function fixCacheability(AccountInterface $acting_user, CacheableDependencyInterface $cacheability): CacheableDependencyInterface {
    if ((int) $acting_user->id() !== (int) $this->currentUser->id()) {
      $cache_contexts = array_reduce($cacheability->getCacheContexts(), static function (array $carry, string $item) use ($acting_user): array {
        if (str_starts_with($item, 'user.node_grants')) {
          $carry[] = NodeGrantsByActingUserCacheContext::generateContextIdFromUserNodeGrantsContextId($item, (int) $acting_user->id());
        }
        else {
          $carry[] = $item;
        }

        return $carry;
      }, []);

      if (!empty(array_diff($cacheability->getCacheContexts(), $cache_contexts))) {
        return (new CacheableMetadata())
          ->addCacheContexts($cache_contexts)
          ->addCacheTags($cacheability->getCacheTags())
          ->setCacheMaxAge($cacheability->getCacheMaxAge());
      }
    }

    return $cacheability;
  }

  /**
   * Collects authored nodes by a user with entity query.
   *
   * @param \Drupal\user\UserInterface $author
   *   The author of nodes.
   * @param \Drupal\Core\Session\AccountInterface $acting_user
   *   The user whose access to the nodes should be checked.
   * @param bool $check_access
   *   Whether to filter out nodes that the $acting_user does not have access
   *   or not.
   *
   * @return int[]|string[]
   *   Array of node IDs,
   */
  private function getAuthoredNodesByUser(UserInterface $author, AccountInterface $acting_user, bool $check_access): array {
    $query = $this->nodeStorage->getQuery();
    $query->accessCheck($check_access);
    // Even if no access checking should be performed, it does not hurt passing
    // this information to hook_query_TAG_alter() implementations.
    $query->addMetaData('account', $acting_user);
    $query->addMetaData('op', $this->entityOperation);
    $query->condition('uid', $author->id());
    return $query->execute();
  }

  /**
   * Executes the query in a render context to catch bubbled cacheability.
   *
   * This was inspired by the JSONAPI module, see reference below.
   *
   * @param callable $callable
   *   A callable.
   * @param \Drupal\Core\Cache\CacheableMetadata $parent_cacheability
   *   The value object to carry the query cacheability.
   *
   * @return mixed
   *   Returns the result of the callback.
   *
   * @see node_query_node_access_alter()
   * @see https://www.drupal.org/project/drupal/issues/2557815
   * @see https://www.drupal.org/project/drupal/issues/2794385
   * @see \Drupal\jsonapi\Controller\EntityResource::executeQueryInRenderContext()
   * @todo Remove this after https://www.drupal.org/project/drupal/issues/3028976 is fixed.
   */
  private function executeCallableInRenderContext(callable $callable, CacheableMetadata $parent_cacheability): mixed {
    $context = new RenderContext();
    $results = $this->renderer->executeInRenderContext($context, static function () use ($callable) {
      return $callable();
    });
    if (!$context->isEmpty()) {
      $parent_cacheability->addCacheableDependency($context->pop());
    }
    return $results;
  }

  /**
   * Allows checking the default entity operation access that is checked.
   *
   * @param string $entity_operation
   *   The entity operation access to be checked. Possible values: view, update
   *   or delete.
   */
  public function setEntityOperation(string $entity_operation): void {
    if (!in_array($entity_operation, self::SUPPORTED_ENTITY_OPERATIONS)) {
      throw new \LogicException(sprintf('The entity operation must be one of these: %s.', implode(', ', self::SUPPORTED_ENTITY_OPERATIONS)));
    }
    $this->entityOperation = $entity_operation;
  }

  /**
   * Allows changing the default entity load batch size.
   *
   * @param int $entity_load_batch_size
   *   The amount of entities to be loaded at once to verify entity access.
   */
  public function setEntityLoadBatchSize(int $entity_load_batch_size): void {
    if ($entity_load_batch_size <= 0) {
      throw new \LogicException('The entity load batch size must be greater than zero.');
    }
    $this->entityLoadBatchSize = $entity_load_batch_size;
  }

}
