<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2023 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

namespace Drupal\Tests\view_usernames_node_author\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\view_usernames_render_cache_tester\Traits\RenderCacheTestTrait;
use Drupal\node\NodeInterface;
use Drupal\user\UserInterface;

/**
 * Smoke tests for render caching.
 */
final class RenderCacheTest extends BrowserTestBase {

  use RenderCacheTestTrait;
  use NodeCreationTrait;
  use ContentTypeCreationTrait;

  private const USERNAME_AUTHOR = 'author';

  private const USERNAME_VIEWER = 'viewer';

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'view_usernames_render_cache_tester',
    'view_usernames_node_author',
    'view_usernames_node_author_node_access',
  ];

  /**
   * A simple registered user.
   *
   * @var \Drupal\user\UserInterface
   */
  private UserInterface $author;

  /**
   * A user that can be viewed by anyone until it has a special role.
   *
   * @var \Drupal\user\UserInterface
   */
  private UserInterface $viewer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->author = $this->createUser([], self::USERNAME_AUTHOR);
    $this->viewer = $this->createUser([], self::USERNAME_VIEWER);

    $this->createContentType(['type' => 'page', 'name' => 'Basic page']);
  }

  /**
   * Tests caching and cache invalidation on pages that list usernames.
   */
  public function testPageCacheabilityWithMultipleUsers(): void {
    $this->visitUsernameList();
    $this->assertUserNameVisibility(FALSE, FALSE, FALSE);

    $this->drupalLogin($this->viewer);
    $this->visitUsernameList();
    $this->assertSession()->responseHeaderEquals('X-Drupal-Dynamic-Cache', 'MISS');
    $this->assertUserNameVisibility(FALSE, TRUE, FALSE);
    $this->visitUsernameList();
    $this->assertSession()->responseHeaderEquals('X-Drupal-Dynamic-Cache', 'HIT');

    $node = $this->drupalCreateNode([
      'type' => 'page',
      'uid' => $this->author->id(),
      'status' => NodeInterface::PUBLISHED,
    ]);

    $this->visitUsernameList();
    $this->assertSession()->responseHeaderEquals('X-Drupal-Dynamic-Cache', 'MISS');
    $this->assertUserNameVisibility(TRUE, TRUE, FALSE);
    $this->visitUsernameList();
    $this->assertSession()->responseHeaderEquals('X-Drupal-Dynamic-Cache', 'HIT');

    $node->setUnpublished()->save();

    $this->visitUsernameList();
    $this->assertSession()->responseHeaderEquals('X-Drupal-Dynamic-Cache', 'MISS');
    $this->assertUserNameVisibility(FALSE, TRUE, FALSE);
  }

  /**
   * Tests cacheability on pages that outputs a username.
   */
  public function testPageCacheabilityWithOneUsername(): void {
    $this->drupalLogin($this->viewer);
    $this->assertOnUsernameViewPages(
      $this->author,
      fn() => $this->assertSession()->pageTextNotContains(self::USERNAME_AUTHOR)
    );

    $this->drupalCreateNode([
      'type' => 'page',
      'uid' => $this->author->id(),
      'status' => NodeInterface::PUBLISHED,
    ]);
    $this->assertOnUsernameViewPages(
      $this->author,
      fn() => $this->assertSession()->pageTextContains(self::USERNAME_AUTHOR)
    );

    \view_usernames_node_author_node_access_grant_exclusive_access_via_node_grants((int) $this->author->id());
    $this->assertOnUsernameViewPages(
      $this->author,
      fn() => $this->assertSession()->pageTextNotContains(self::USERNAME_AUTHOR)
    );

    \view_usernames_node_author_node_access_grant_access_via_node_access((int) $this->viewer->id());
    $this->assertOnUsernameViewPages(
      $this->author,
      fn() => $this->assertSession()->pageTextContains(self::USERNAME_AUTHOR)
    );

    \view_usernames_node_author_node_access_grant_exclusive_access_via_node_grants(NULL);
    \view_usernames_node_author_node_access_grant_access_via_node_access(NULL);
    $this->assertOnUsernameViewPages(
      $this->author,
      fn() => $this->assertSession()->pageTextContains(self::USERNAME_AUTHOR)
    );

    \view_usernames_node_author_node_access_grant_exclusive_access_via_node_grants((int) $this->viewer->id());
    \view_usernames_node_author_node_access_revoke_access_via_node_access((int) $this->viewer->id());
    $this->assertOnUsernameViewPages(
      $this->author,
      fn() => $this->assertSession()->pageTextNotContains(self::USERNAME_AUTHOR)
    );
  }

  /**
   * Tests cacheability on pages that outputs a username.
   */
  public function testPageCacheabilityWithViewAs(): void {
    // We MUST log in with a user because cache invalidation does not work
    // with a non-logged in user. See README.md.
    $somebody = $this->createUser([], 'Somebody');
    $this->drupalLogin($somebody);

    $this->visitUsernameViewAsUser($this->author, $this->viewer);
    $this->assertSession()->pageTextNotContains(self::USERNAME_AUTHOR);

    $this->drupalCreateNode([
      'type' => 'page',
      'uid' => $this->author->id(),
      'status' => NodeInterface::PUBLISHED,
    ]);
    $this->visitUsernameViewAsUser($this->author, $this->viewer);
    $this->assertSession()->pageTextContains(self::USERNAME_AUTHOR);

    \view_usernames_node_author_node_access_grant_exclusive_access_via_node_grants((int) $this->author->id());
    $this->visitUsernameViewAsUser($this->author, $this->viewer);
    $this->assertSession()->pageTextNotContains(self::USERNAME_AUTHOR);

    \view_usernames_node_author_node_access_grant_access_via_node_access((int) $this->viewer->id());
    $this->visitUsernameViewAsUser($this->author, $this->viewer);
    $this->assertSession()->pageTextContains(self::USERNAME_AUTHOR);

    \view_usernames_node_author_node_access_grant_exclusive_access_via_node_grants(NULL);
    \view_usernames_node_author_node_access_grant_access_via_node_access(NULL);
    $this->visitUsernameViewAsUser($this->author, $this->viewer);
    $this->assertSession()->pageTextContains(self::USERNAME_AUTHOR);

    \view_usernames_node_author_node_access_grant_exclusive_access_via_node_grants((int) $this->viewer->id());
    \view_usernames_node_author_node_access_revoke_access_via_node_access((int) $this->viewer->id());
    $this->visitUsernameViewAsUser($this->author, $this->viewer);
    $this->assertSession()->pageTextNotContains(self::USERNAME_AUTHOR);
  }

  /**
   * Asserts the expected usernames are visible on the page.
   *
   * @param bool $is_author_user_visible
   *   Whether the simple user is visible or not.
   * @param bool $is_viewer_user_visible
   *   Whether the special user is visible or not.
   * @param bool $is_admin_visible
   *   Whether the admin (uid1) is visible or not.
   */
  private function assertUserNameVisibility(bool $is_author_user_visible, bool $is_viewer_user_visible, bool $is_admin_visible): void {
    // Anonymous is always visible.
    $this->assertSession()->pageTextContains('Anonymous');
    $is_author_user_visible
        ? $this->assertSession()->pageTextContains(self::USERNAME_AUTHOR)
        : $this->assertSession()->pageTextNotContains(self::USERNAME_AUTHOR);
    $is_viewer_user_visible
        ? $this->assertSession()->pageTextContains(self::USERNAME_VIEWER)
        : $this->assertSession()->pageTextNotContains(self::USERNAME_VIEWER);
    $is_admin_visible
        ? $this->assertSession()->pageTextContains('admin')
        : $this->assertSession()->pageTextNotContains('admin');
  }

}
