<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2023 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

namespace Drupal\Tests\view_usernames_node_author\Kernel;

use Drupal\node\NodeInterface;

/**
 * Tests implementation in view_usernames_node_author_node_access.
 *
 * In addition, it covers special cases of how hook_node_grants() and
 * hook_node_access() works together.
 *
 * @covers \view_usernames_node_author_node_access_node_grants()
 * @covers \view_usernames_node_author_node_access_node_access_records()
 * @covers \view_usernames_node_author_node_access_node_access()
 */
final class NodeAccessTest extends NodeAuthorViewUsernameAccessDeciderTestBase {

  /**
   * A node created by the author.
   *
   * @var \Drupal\node\NodeInterface
   */
  private NodeInterface $node;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->enableViewUsernamesNodeAuthorNodeAccessModule();

    $this->node = $this->createNode([
      'type' => 'page',
      'uid' => $this->author->id(),
      'status' => NodeInterface::PUBLISHED,
    ]);
  }

  /**
   * Tests viewer can view the authors published node by default.
   */
  public function testViewerCanViewAuthorsPublishedNodeByDefault(): void {
    $this->assertTrue($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));
  }

  /**
   * Tests viewer cannot view the authors unpublished node by default.
   */
  public function testViewerCannotViewAuthorsUnPublishedNodeByDefault(): void {
    $this->node->setUnpublished()->save();
    $this->assertFalse($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));
  }

  /**
   * Test our hook_node_grants() implementation can override access.
   */
  public function testViewerCanViewAuthorsUnPublishedNodeWithIndividualAccessGrantedByHookNodeAccessGrants(): void {
    $this->node->setUnpublished()->save();
    $this->grantExclusiveAccessViaHookNodeGrants((int) $this->viewer->id());
    $this->assertTrue($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));
  }

  /**
   * Test our hook_node_access() implementation can override access.
   */
  public function testViewerCanViewAuthorsUnPublishedNodeWithIndividualAccessGrantedByHookNodeAccess(): void {
    $this->node->setUnpublished()->save();
    $this->grantUserAccessViaHookNodeAccess((int) $this->viewer->id());
    $this->assertTrue($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));
  }

  /**
   * Hook_node_access() can change access granted by hook_node_grants().
   */
  public function testHookNodeAccessCanOverrideAccessGrantedByHookNodeGrants(): void {
    $this->node->setUnpublished()->save();
    $this->assertFalse($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));

    $this->grantExclusiveAccessViaHookNodeGrants((int) $this->viewer->id());
    $this->assertTrue($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));

    $this->revokeUserAccessViaHookNodeAccess((int) $this->viewer->id());
    $this->assertFalse($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));
  }

  /**
   * Hook_node_grants() cannot change access granted by hook_node_access().
   */
  public function testHookNodeAccessGrantsCanNotOverrideAccessGrantedByHookNodeAccess(): void {
    $this->node->setUnpublished()->save();
    $this->assertFalse($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));

    $this->grantUserAccessViaHookNodeAccess((int) $this->viewer->id());
    $this->assertTrue($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));

    $somebody_else = $this->createUser(['access content'], 'somebody else');
    $this->grantExclusiveAccessViaHookNodeGrants((int) $somebody_else->id());
    $this->assertTrue($this->nodeAccessHandler->access($this->node, 'view', $this->viewer));
  }

}
