<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2023 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */


namespace Drupal\Tests\view_usernames_node_author\Kernel;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\node\NodeInterface;
use Drupal\user\RoleInterface;
use Drupal\view_usernames_node_author\NodeAuthorViewUsernameAccessDecider;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Tests access to a node author's username.
 *
 * @covers \Drupal\view_usernames_node_author\NodeAuthorViewUsernameAccessDecider
 *
 * @todo Cover \Drupal\view_usernames_node_author\NodeAuthorViewUsernameAccessDecider::setEntityLoadBatchSize()
 */
final class NodeAuthorUsernameAccessDeciderTest extends NodeAuthorViewUsernameAccessDeciderTestBase {

  /**
   * {@inheritdoc}
   */
  public function register(ContainerBuilder $container): void {
    parent::register($container);

    $this->container->register('view_usernames_node_author_test.view_username_access_decider.node_author_by_update_access', NodeAuthorViewUsernameAccessDecider::class)
      ->addArgument(new Reference('entity_type.manager'))
      ->addArgument(new Reference('renderer'))
      ->addArgument(new Reference('current_user'))
      ->addMethodCall('setEntityOperation', ['update'])
      ->addTag('view_username_access_decider');
  }

  /**
   * Tests that a published node's author's name is visible to everyone.
   *
   * This test also covers the case when there is no active
   * hook_node_access_grants() implementation.
   */
  public function testPublishedNodesAuthorNameIsVisibleToEveryone(): void {
    $this->assertFalse($this->container->get('module_handler')->hasImplementations('node_grants'));

    $anonymous = $this->container->get('entity_type.manager')->getStorage('user')->load(0);

    $this->assertUserCouldNotAccessOtherUsersUsername($this->author, $anonymous);
    $this->assertUserCouldNotAccessOtherUsersUsername($this->author, $this->viewer);

    $node = $this->createNode([
      'type' => 'page',
      'uid' => $this->author->id(),
      'status' => NodeInterface::PUBLISHED,
    ]);
    $this->assertUserCanAccessOtherUsersUsername($this->author, $anonymous);
    $this->assertUserCanAccessOtherUsersUsername($this->author, $this->viewer);

    $node->setUnpublished()->save();
    $this->assertUserCannotAccessOtherUsersUsername($this->author, $anonymous);
    $this->assertUserCannotAccessOtherUsersUsername($this->author, $this->viewer);
  }

  /**
   * Tests username access when different node access impl. are involved.
   *
   * When there is at least one hook_node_access_grants() implementation active
   * then system behaves differently.
   *
   * @see \Drupal\node\Cache\NodeAccessGrantsCacheContext::getCacheableMetadata()
   * @see \node_access_view_all_nodes()
   */
  public function testIntegrationWithNodeAccessImplementations(): void {
    $anonymous = $this->container->get('entity_type.manager')->getStorage('user')->load(0);
    $this->enableViewUsernamesNodeAuthorNodeAccessModule();
    $this->assertTrue($this->container->get('module_handler')->hasImplementations('node_grants'));

    $this->createNode([
      'type' => 'page',
      'uid' => $this->author->id(),
      'status' => NodeInterface::NOT_PUBLISHED,
    ]);
    $this->assertUserCouldNotAccessOtherUsersUsername($this->author, $anonymous);
    $this->assertUserCouldNotAccessOtherUsersUsername($this->author, $this->viewer);

    $this->grantExclusiveAccessViaHookNodeGrants((int) $this->viewer->id());
    $this->assertUserCannotAccessOtherUsersUsername($this->author, $anonymous);
    $this->assertUserCanAccessOtherUsersUsername($this->author, $this->viewer);

    $this->revokeUserAccessViaHookNodeAccess((int) $this->viewer->id());
    $this->assertUserCanNotAccessOtherUsersUsername($this->author, $this->viewer);
  }

  /**
   * Tests access can be granted based on node update access in custom code.
   *
   * @see static::register()
   */
  public function testUsernameAccessCanBeGrantedBasedOnNodeUpdateAccessViaCustomDecider(): void {
    $this->enableViewUsernamesNodeAuthorNodeAccessModule();
    $this->createNode([
      'type' => 'page',
      'uid' => $this->author->id(),
      'status' => NodeInterface::NOT_PUBLISHED,
    ]);
    $this->assertUserCouldNotAccessOtherUsersUsername($this->author, $this->viewer);

    user_role_grant_permissions(RoleInterface::AUTHENTICATED_ID, ['edit any page content']);
    $this->flushAccessHandlersStaticCache();
    $this->assertUserCanAccessOtherUsersUsername($this->author, $this->viewer);
  }

}
