<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2023 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */


namespace Drupal\Tests\view_usernames_node_author\Kernel;

use Drupal\Core\Entity\EntityAccessControlHandlerInterface;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\Tests\view_usernames\Kernel\UsernameAccessTestKernelTestBase;
use Drupal\user\RoleInterface;
use Drupal\user\UserInterface;

/**
 * Base class for username access tests that involves node access.
 */
abstract class NodeAuthorViewUsernameAccessDeciderTestBase extends UsernameAccessTestKernelTestBase {

  use ContentTypeCreationTrait;
  use UserCreationTrait;
  use NodeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    // Next three items are required for creating the test node type and test
    // nodes.
    'field',
    'text',
    'filter',
    'node',
    'view_usernames',
    'view_usernames_node_author',
  ];

  /**
   * The user that when it is needed an author of a node.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $author;

  /**
   * The user who tries to access the author's username.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $viewer;

  /**
   * The node access handler.
   *
   * @var \Drupal\Core\Entity\EntityAccessControlHandlerInterface
   */
  protected EntityAccessControlHandlerInterface $nodeAccessHandler;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('node', 'node_access');
    $this->installEntitySchema('node');
    $this->installConfig('filter');
    $this->installConfig('node');

    $this->createContentType(['type' => 'page', 'name' => 'Basic page']);

    $this->nodeAccessHandler = $this->container->get('entity_type.manager')->getAccessControlHandler('node');

    $this->author = $this->createUser([], 'Author');
    $this->viewer = $this->createUser([], 'Viewer');

    user_role_grant_permissions(RoleInterface::ANONYMOUS_ID, ['access content']);
    user_role_grant_permissions(RoleInterface::AUTHENTICATED_ID, ['access content']);
  }

  /**
   * Enables the view_usernames_node_author_node_access module.
   */
  final protected function enableViewUsernamesNodeAuthorNodeAccessModule(): void {
    $this->enableModules(['view_usernames_node_author_node_access']);
    \node_access_rebuild();
    // We need the updated node access handler with newly enabled hooks.
    $this->nodeAccessHandler = $this->container->get('entity_type.manager')->getAccessControlHandler('node');
  }

  /**
   * {@inheritdoc}
   */
  protected function postAssertUsernameAccess(): void {
    $this->accessHandler->resetCache();
    $this->nodeAccessHandler->resetCache();
  }

  /**
   * Flushes access handler's static cache in tests.
   *
   * This is a MUST if one test performs multiple assertions on access.
   */
  final protected function flushAccessHandlersStaticCache(): void {
    $this->accessHandler->resetCache();
    $this->nodeAccessHandler->resetCache();
  }

  /**
   * Grants exclusive access to all nodes via hook_node_grants().
   *
   * @param int|null $uid
   *   The UID of a user or null if the previous grant should be deleted.
   */
  final protected function grantExclusiveAccessViaHookNodeGrants(?int $uid): void {
    view_usernames_node_author_node_access_grant_exclusive_access_via_node_grants($uid);
    // Because we changed node grants we have to refresh the previously
    // cached version of node access handler.
    $this->nodeAccessHandler = $this->container->get('entity_type.manager')->getAccessControlHandler('node');
    $this->flushAccessHandlersStaticCache();
  }

  /**
   * Grant access to a user for a node via hook_node_access().
   *
   * @param int|null $uid
   *   The UID of a user or null if the previous grant should be deleted.
   */
  final protected function grantUserAccessViaHookNodeAccess(?int $uid): void {
    \view_usernames_node_author_node_access_grant_access_via_node_access($uid);
    $this->flushAccessHandlersStaticCache();
  }

  /**
   * Revokes a user's access to a node via hook_node_access().
   *
   * @param int|null $uid
   *   The UID of a user or null if the previous restriction should be deleted.
   */
  final protected function revokeUserAccessViaHookNodeAccess(?int $uid): void {
    view_usernames_node_author_node_access_revoke_access_via_node_access($uid);
    $this->flushAccessHandlersStaticCache();
  }

}
