<?php

namespace Drupal\viewer\Hook;

use Drupal\Component\Utility\Random;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Hook implementations for the viewer module.
 */
class ViewerHooks {

  use StringTranslationTrait;

  /**
   * Constructs a ViewerHooks object.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $currentRouteMatch
   *   The current route match service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(
    protected DateFormatterInterface $dateFormatter,
    protected RequestStack $requestStack,
    protected ModuleHandlerInterface $moduleHandler,
    protected RouteMatchInterface $currentRouteMatch,
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Implements hook_page_attachments().
   */
  #[Hook('page_attachments')]
  public function pageAttachments(array &$attachments): void {
    $base_url = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();
    $module_path = $this->moduleHandler->getModule('viewer')->getPath();
    $attachments['#attached']['drupalSettings']['viewer']['path'] = $base_url . '/' . $module_path . '/';
  }

  /**
   * Implements hook_mail().
   */
  #[Hook('mail')]
  public function mail(string $key, array &$message, array $params): void {
    if ($key == 'notification') {
      $message['from'] = $this->configFactory->get('system.site')->get('mail');
      $message['subject'] = $params['subject'];
      $message['body'][] = $params['message'];
    }
  }

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'viewer_table' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'settings' => [],
          'last_import' => NULL,
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_footable' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'settings' => [],
          'last_import' => NULL,
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_spreadsheet' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'settings' => [],
          'configuration' => [],
          'last_import' => NULL,
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_pdfjs' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'settings' => [],
          'last_import' => NULL,
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_tabs' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'settings' => [],
          'tabs' => [],
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_accordion' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'settings' => [],
          'accordion' => [],
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_chartjs' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'wrapper' => NULL,
          'type' => NULL,
          'labels' => NULL,
          'settings' => [],
          'last_import' => NULL,
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_apexchart' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'wrapper' => NULL,
          'type' => NULL,
          'labels' => NULL,
          'settings' => [],
          'last_import' => NULL,
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_datatables' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'settings' => [],
          'last_import' => NULL,
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'viewer_fullcalendar' => [
        'variables' => [
          'uuid' => NULL,
          'viewer' => NULL,
          'settings' => [],
          'configuration' => [],
          'last_import' => NULL,
        ],
        'base hook' => 'viewer_plugin',
        'preprocess functions' => ['_viewer_preprocess_viewer_plugin'],
      ],
      'page__admin__structure__viewers__iframe_src' => [
        'template' => 'page--admin--structure--viewers--iframe-src',
        'base hook' => 'page',
      ],
    ];
  }

  /**
   * Implements hook_preprocess_html().
   */
  #[Hook('preprocess_html')]
  public function preprocessHtml(array &$variables): void {
    // Hide admin toolbar from the preview page.
    if ($this->currentRouteMatch->getRouteName() == 'entity.viewer.iframe_preview_src') {
      $variables['page_top']['toolbar']['#access'] = FALSE;
    }
  }

  /**
   * Implements hook_theme_suggestions_HOOK_alter() for viewer_plugin.
   */
  #[Hook('theme_suggestions_viewer_plugin_alter')]
  public function themeSuggestionsViewerPluginAlter(array &$suggestions, array $variables): void {
    $suggestions[] = 'viewer_' . $variables['viewer']->getViewerPluginId() . '__' . $variables['viewer']->id();
  }

  /**
   * Implements hook_cron().
   */
  #[Hook('cron')]
  public function cron(): void {
    \Drupal::service('viewer.cron')->processQueue();
  }

  /**
   * Get import frequencies.
   *
   * @return array
   *   Array of import frequency options.
   */
  public function getImportFrequencies(): array {
    $options = [
      300, 600, 900, 1800, 3600, 10800,
      21600, 43200, 86400, 604800, 2629743, 31556926,
    ];
    return [0 => $this->t('Manual')]
      + array_map([$this->dateFormatter, 'formatInterval'],
      array_combine($options, $options));
  }

  /**
   * Preprocess function for viewer plugin templates.
   *
   * @param array $variables
   *   The variables array.
   */
  public function preprocessViewerPlugin(array &$variables): void {
    $variables['title'] = !empty($variables['settings']['title']) ? $variables['settings']['title'] : '';
    $variables['subtitle'] = !empty($variables['settings']['subtitle']) ? $variables['settings']['subtitle'] : '';
    $variables['header_summary'] = !empty($variables['settings']['header_summary']) ? $variables['settings']['header_summary'] : '';
    $variables['footer_summary'] = !empty($variables['settings']['footer_summary']) ? $variables['settings']['footer_summary'] : '';
    if (!empty($variables['settings']['last_import_format'])) {
      $variables['last_import_formatted'] = $this->dateFormatter
        ->format($variables['last_import'], $variables['settings']['last_import_format']);
      $variables['last_import_position'] = !empty($variables['settings']['last_import_position']) ? $variables['settings']['last_import_position'] : '';
      $variables['last_import_output'] = str_replace('@date', $variables['last_import_formatted'], $variables['settings']['last_import']);
    }
    $variables['random'] = md5((new Random())->string(8, TRUE));
  }

}
